<?php

namespace PHPMaker2024\fibre;

use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use Psr\Http\Message\ServerRequestInterface as Request;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Container\ContainerInterface;
use Slim\Routing\RouteCollectorProxy;
use Slim\App;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Closure;

/**
 * Page class
 */
class ServiceRequestSearch extends ServiceRequest
{
    use MessagesTrait;

    // Page ID
    public $PageID = "search";

    // Project ID
    public $ProjectID = PROJECT_ID;

    // Page object name
    public $PageObjName = "ServiceRequestSearch";

    // View file path
    public $View = null;

    // Title
    public $Title = null; // Title for <title> tag

    // Rendering View
    public $RenderingView = false;

    // CSS class/style
    public $CurrentPageName = "servicerequestsearch";

    // Page headings
    public $Heading = "";
    public $Subheading = "";
    public $PageHeader;
    public $PageFooter;

    // Page layout
    public $UseLayout = true;

    // Page terminated
    private $terminated = false;

    // Page heading
    public function pageHeading()
    {
        global $Language;
        if ($this->Heading != "") {
            return $this->Heading;
        }
        if (method_exists($this, "tableCaption")) {
            return $this->tableCaption();
        }
        return "";
    }

    // Page subheading
    public function pageSubheading()
    {
        global $Language;
        if ($this->Subheading != "") {
            return $this->Subheading;
        }
        if ($this->TableName) {
            return $Language->phrase($this->PageID);
        }
        return "";
    }

    // Page name
    public function pageName()
    {
        return CurrentPageName();
    }

    // Page URL
    public function pageUrl($withArgs = true)
    {
        $route = GetRoute();
        $args = RemoveXss($route->getArguments());
        if (!$withArgs) {
            foreach ($args as $key => &$val) {
                $val = "";
            }
            unset($val);
        }
        return rtrim(UrlFor($route->getName(), $args), "/") . "?";
    }

    // Show Page Header
    public function showPageHeader()
    {
        $header = $this->PageHeader;
        $this->pageDataRendering($header);
        if ($header != "") { // Header exists, display
            echo '<div id="ew-page-header">' . $header . '</div>';
        }
    }

    // Show Page Footer
    public function showPageFooter()
    {
        $footer = $this->PageFooter;
        $this->pageDataRendered($footer);
        if ($footer != "") { // Footer exists, display
            echo '<div id="ew-page-footer">' . $footer . '</div>';
        }
    }

    // Set field visibility
    public function setVisibility()
    {
        $this->id->setVisibility();
        $this->status->setVisibility();
        $this->buyer_id->setVisibility();
        $this->product_ref->setVisibility();
        $this->current_reading->setVisibility();
        $this->product_status->setVisibility();
        $this->_error->setVisibility();
        $this->last_serve->setVisibility();
        $this->last_serve_by->setVisibility();
        $this->query_by->setVisibility();
        $this->service_executive->setVisibility();
        $this->Ins_service_executive->setVisibility();
        $this->operator->setVisibility();
        $this->datetime->setVisibility();
        $this->o_operator->setVisibility();
        $this->o_datetime->setVisibility();
        $this->rid_1->Visible = false;
        $this->master_reading->setVisibility();
    }

    // Constructor
    public function __construct()
    {
        parent::__construct();
        global $Language, $DashboardReport, $DebugTimer, $UserTable;
        $this->TableVar = 'service_request';
        $this->TableName = 'service_request';

        // Table CSS class
        $this->TableClass = "table table-striped table-bordered table-hover table-sm ew-desktop-table ew-search-table";

        // Initialize
        $GLOBALS["Page"] = &$this;

        // Language object
        $Language = Container("app.language");

        // Table object (service_request)
        if (!isset($GLOBALS["service_request"]) || $GLOBALS["service_request"]::class == PROJECT_NAMESPACE . "service_request") {
            $GLOBALS["service_request"] = &$this;
        }

        // Table name (for backward compatibility only)
        if (!defined(PROJECT_NAMESPACE . "TABLE_NAME")) {
            define(PROJECT_NAMESPACE . "TABLE_NAME", 'service_request');
        }

        // Start timer
        $DebugTimer = Container("debug.timer");

        // Debug message
        LoadDebugMessage();

        // Open connection
        $GLOBALS["Conn"] ??= $this->getConnection();

        // User table object
        $UserTable = Container("usertable");
    }

    // Get content from stream
    public function getContents(): string
    {
        global $Response;
        return $Response?->getBody() ?? ob_get_clean();
    }

    // Is lookup
    public function isLookup()
    {
        return SameText(Route(0), Config("API_LOOKUP_ACTION"));
    }

    // Is AutoFill
    public function isAutoFill()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autofill");
    }

    // Is AutoSuggest
    public function isAutoSuggest()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autosuggest");
    }

    // Is modal lookup
    public function isModalLookup()
    {
        return $this->isLookup() && SameText(Post("ajax"), "modal");
    }

    // Is terminated
    public function isTerminated()
    {
        return $this->terminated;
    }

    /**
     * Terminate page
     *
     * @param string $url URL for direction
     * @return void
     */
    public function terminate($url = "")
    {
        if ($this->terminated) {
            return;
        }
        global $TempImages, $DashboardReport, $Response;

        // Page is terminated
        $this->terminated = true;

        // Page Unload event
        if (method_exists($this, "pageUnload")) {
            $this->pageUnload();
        }
        DispatchEvent(new PageUnloadedEvent($this), PageUnloadedEvent::NAME);
        if (!IsApi() && method_exists($this, "pageRedirecting")) {
            $this->pageRedirecting($url);
        }

        // Close connection
        CloseConnections();

        // Return for API
        if (IsApi()) {
            $res = $url === true;
            if (!$res) { // Show response for API
                $ar = array_merge($this->getMessages(), $url ? ["url" => GetUrl($url)] : []);
                WriteJson($ar);
            }
            $this->clearMessages(); // Clear messages for API request
            return;
        } else { // Check if response is JSON
            if (WithJsonResponse()) { // With JSON response
                $this->clearMessages();
                return;
            }
        }

        // Go to URL if specified
        if ($url != "") {
            if (!Config("DEBUG") && ob_get_length()) {
                ob_end_clean();
            }

            // Handle modal response
            if ($this->IsModal) { // Show as modal
                $pageName = GetPageName($url);
                $result = ["url" => GetUrl($url), "modal" => "1"];  // Assume return to modal for simplicity
                if (!SameString($pageName, GetPageName($this->getListUrl()))) { // Not List page
                    $result["caption"] = $this->getModalCaption($pageName);
                    $result["view"] = SameString($pageName, "servicerequestview"); // If View page, no primary button
                } else { // List page
                    $result["error"] = $this->getFailureMessage(); // List page should not be shown as modal => error
                    $this->clearFailureMessage();
                }
                WriteJson($result);
            } else {
                SaveDebugMessage();
                Redirect(GetUrl($url));
            }
        }
        return; // Return to controller
    }

    // Get records from result set
    protected function getRecordsFromRecordset($rs, $current = false)
    {
        $rows = [];
        if (is_object($rs)) { // Result set
            while ($row = $rs->fetch()) {
                $this->loadRowValues($row); // Set up DbValue/CurrentValue
                $row = $this->getRecordFromArray($row);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        } elseif (is_array($rs)) {
            foreach ($rs as $ar) {
                $row = $this->getRecordFromArray($ar);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        }
        return $rows;
    }

    // Get record from array
    protected function getRecordFromArray($ar)
    {
        $row = [];
        if (is_array($ar)) {
            foreach ($ar as $fldname => $val) {
                if (array_key_exists($fldname, $this->Fields) && ($this->Fields[$fldname]->Visible || $this->Fields[$fldname]->IsPrimaryKey)) { // Primary key or Visible
                    $fld = &$this->Fields[$fldname];
                    if ($fld->HtmlTag == "FILE") { // Upload field
                        if (EmptyValue($val)) {
                            $row[$fldname] = null;
                        } else {
                            if ($fld->DataType == DataType::BLOB) {
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . $fld->Param . "/" . rawurlencode($this->getRecordKeyValue($ar))));
                                $row[$fldname] = ["type" => ContentType($val), "url" => $url, "name" => $fld->Param . ContentExtension($val)];
                            } elseif (!$fld->UploadMultiple || !ContainsString($val, Config("MULTIPLE_UPLOAD_SEPARATOR"))) { // Single file
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $val)));
                                $row[$fldname] = ["type" => MimeContentType($val), "url" => $url, "name" => $val];
                            } else { // Multiple files
                                $files = explode(Config("MULTIPLE_UPLOAD_SEPARATOR"), $val);
                                $ar = [];
                                foreach ($files as $file) {
                                    $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                        "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $file)));
                                    if (!EmptyValue($file)) {
                                        $ar[] = ["type" => MimeContentType($file), "url" => $url, "name" => $file];
                                    }
                                }
                                $row[$fldname] = $ar;
                            }
                        }
                    } else {
                        $row[$fldname] = $val;
                    }
                }
            }
        }
        return $row;
    }

    // Get record key value from array
    protected function getRecordKeyValue($ar)
    {
        $key = "";
        if (is_array($ar)) {
            $key .= @$ar['id'];
        }
        return $key;
    }

    /**
     * Hide fields for add/edit
     *
     * @return void
     */
    protected function hideFieldsForAddEdit()
    {
        if ($this->isAdd() || $this->isCopy() || $this->isGridAdd()) {
            $this->id->Visible = false;
        }
    }

    // Lookup data
    public function lookup(array $req = [], bool $response = true)
    {
        global $Language, $Security;

        // Get lookup object
        $fieldName = $req["field"] ?? null;
        if (!$fieldName) {
            return [];
        }
        $fld = $this->Fields[$fieldName];
        $lookup = $fld->Lookup;
        $name = $req["name"] ?? "";
        if (ContainsString($name, "query_builder_rule")) {
            $lookup->FilterFields = []; // Skip parent fields if any
        }

        // Get lookup parameters
        $lookupType = $req["ajax"] ?? "unknown";
        $pageSize = -1;
        $offset = -1;
        $searchValue = "";
        if (SameText($lookupType, "modal") || SameText($lookupType, "filter")) {
            $searchValue = $req["q"] ?? $req["sv"] ?? "";
            $pageSize = $req["n"] ?? $req["recperpage"] ?? 10;
        } elseif (SameText($lookupType, "autosuggest")) {
            $searchValue = $req["q"] ?? "";
            $pageSize = $req["n"] ?? -1;
            $pageSize = is_numeric($pageSize) ? (int)$pageSize : -1;
            if ($pageSize <= 0) {
                $pageSize = Config("AUTO_SUGGEST_MAX_ENTRIES");
            }
        }
        $start = $req["start"] ?? -1;
        $start = is_numeric($start) ? (int)$start : -1;
        $page = $req["page"] ?? -1;
        $page = is_numeric($page) ? (int)$page : -1;
        $offset = $start >= 0 ? $start : ($page > 0 && $pageSize > 0 ? ($page - 1) * $pageSize : 0);
        $userSelect = Decrypt($req["s"] ?? "");
        $userFilter = Decrypt($req["f"] ?? "");
        $userOrderBy = Decrypt($req["o"] ?? "");
        $keys = $req["keys"] ?? null;
        $lookup->LookupType = $lookupType; // Lookup type
        $lookup->FilterValues = []; // Clear filter values first
        if ($keys !== null) { // Selected records from modal
            if (is_array($keys)) {
                $keys = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $keys);
            }
            $lookup->FilterFields = []; // Skip parent fields if any
            $lookup->FilterValues[] = $keys; // Lookup values
            $pageSize = -1; // Show all records
        } else { // Lookup values
            $lookup->FilterValues[] = $req["v0"] ?? $req["lookupValue"] ?? "";
        }
        $cnt = is_array($lookup->FilterFields) ? count($lookup->FilterFields) : 0;
        for ($i = 1; $i <= $cnt; $i++) {
            $lookup->FilterValues[] = $req["v" . $i] ?? "";
        }
        $lookup->SearchValue = $searchValue;
        $lookup->PageSize = $pageSize;
        $lookup->Offset = $offset;
        if ($userSelect != "") {
            $lookup->UserSelect = $userSelect;
        }
        if ($userFilter != "") {
            $lookup->UserFilter = $userFilter;
        }
        if ($userOrderBy != "") {
            $lookup->UserOrderBy = $userOrderBy;
        }
        return $lookup->toJson($this, $response); // Use settings from current page
    }
    public $FormClassName = "ew-form ew-search-form";
    public $IsModal = false;
    public $IsMobileOrModal = false;

    /**
     * Page run
     *
     * @return void
     */
    public function run()
    {
        global $ExportType, $Language, $Security, $CurrentForm, $SkipHeaderFooter;

        // Is modal
        $this->IsModal = ConvertToBool(Param("modal"));
        $this->UseLayout = $this->UseLayout && !$this->IsModal;

        // Use layout
        $this->UseLayout = $this->UseLayout && ConvertToBool(Param(Config("PAGE_LAYOUT"), true));

        // View
        $this->View = Get(Config("VIEW"));

        // Load user profile
        if (IsLoggedIn()) {
            Profile()->setUserName(CurrentUserName())->loadFromStorage();
        }

        // Create form object
        $CurrentForm = new HttpForm();
        $this->CurrentAction = Param("action"); // Set up current action
        $this->setVisibility();

        // Set lookup cache
        if (!in_array($this->PageID, Config("LOOKUP_CACHE_PAGE_IDS"))) {
            $this->setUseLookupCache(false);
        }

        // Global Page Loading event (in userfn*.php)
        DispatchEvent(new PageLoadingEvent($this), PageLoadingEvent::NAME);

        // Page Load event
        if (method_exists($this, "pageLoad")) {
            $this->pageLoad();
        }

        // Hide fields for add/edit
        if (!$this->UseAjaxActions) {
            $this->hideFieldsForAddEdit();
        }
        // Use inline delete
        if ($this->UseAjaxActions) {
            $this->InlineDelete = true;
        }

        // Set up lookup cache
        $this->setupLookupOptions($this->status);
        $this->setupLookupOptions($this->buyer_id);
        $this->setupLookupOptions($this->product_ref);
        $this->setupLookupOptions($this->product_status);
        $this->setupLookupOptions($this->last_serve_by);
        $this->setupLookupOptions($this->service_executive);

        // Set up Breadcrumb
        $this->setupBreadcrumb();

        // Check modal
        if ($this->IsModal) {
            $SkipHeaderFooter = true;
        }
        $this->IsMobileOrModal = IsMobile() || $this->IsModal;

        // Get action
        $this->CurrentAction = Post("action");
        if ($this->isSearch()) {
            // Build search string for advanced search, remove blank field
            $this->loadSearchValues(); // Get search values
            $srchStr = $this->validateSearch() ? $this->buildAdvancedSearch() : "";
            if ($srchStr != "") {
                $srchStr = "servicerequestlist" . "?" . $srchStr;
                // Do not return Json for UseAjaxActions
                if ($this->IsModal && $this->UseAjaxActions) {
                    $this->IsModal = false;
                }
                $this->terminate($srchStr); // Go to list page
                return;
            }
        }

        // Restore search settings from Session
        if (!$this->hasInvalidFields()) {
            $this->loadAdvancedSearch();
        }

        // Render row for search
        $this->RowType = RowType::SEARCH;
        $this->resetAttributes();
        $this->renderRow();

        // Set LoginStatus / Page_Rendering / Page_Render
        if (!IsApi() && !$this->isTerminated()) {
            // Setup login status
            SetupLoginStatus();

            // Pass login status to client side
            SetClientVar("login", LoginStatus());

            // Global Page Rendering event (in userfn*.php)
            DispatchEvent(new PageRenderingEvent($this), PageRenderingEvent::NAME);

            // Page Render event
            if (method_exists($this, "pageRender")) {
                $this->pageRender();
            }

            // Render search option
            if (method_exists($this, "renderSearchOptions")) {
                $this->renderSearchOptions();
            }
        }
    }

    // Build advanced search
    protected function buildAdvancedSearch()
    {
        $srchUrl = "";
        $this->buildSearchUrl($srchUrl, $this->id); // id
        $this->buildSearchUrl($srchUrl, $this->status); // status
        $this->buildSearchUrl($srchUrl, $this->buyer_id); // buyer_id
        $this->buildSearchUrl($srchUrl, $this->product_ref); // product_ref
        $this->buildSearchUrl($srchUrl, $this->current_reading); // current_reading
        $this->buildSearchUrl($srchUrl, $this->product_status); // product_status
        $this->buildSearchUrl($srchUrl, $this->_error); // error
        $this->buildSearchUrl($srchUrl, $this->last_serve); // last_serve
        $this->buildSearchUrl($srchUrl, $this->last_serve_by); // last_serve_by
        $this->buildSearchUrl($srchUrl, $this->query_by); // query_by
        $this->buildSearchUrl($srchUrl, $this->service_executive); // service_executive
        $this->buildSearchUrl($srchUrl, $this->Ins_service_executive); // Ins_service_executive
        $this->buildSearchUrl($srchUrl, $this->operator); // operator
        $this->buildSearchUrl($srchUrl, $this->datetime); // datetime
        $this->buildSearchUrl($srchUrl, $this->o_operator); // o_operator
        $this->buildSearchUrl($srchUrl, $this->o_datetime); // o_datetime
        $this->buildSearchUrl($srchUrl, $this->master_reading); // master_reading
        if ($srchUrl != "") {
            $srchUrl .= "&";
        }
        $srchUrl .= "cmd=search";
        return $srchUrl;
    }

    // Build search URL
    protected function buildSearchUrl(&$url, $fld, $oprOnly = false)
    {
        global $CurrentForm;
        $wrk = "";
        $fldParm = $fld->Param;
        [
            "value" => $fldVal,
            "operator" => $fldOpr,
            "condition" => $fldCond,
            "value2" => $fldVal2,
            "operator2" => $fldOpr2
        ] = $CurrentForm->getSearchValues($fldParm);
        if (is_array($fldVal)) {
            $fldVal = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal);
        }
        if (is_array($fldVal2)) {
            $fldVal2 = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal2);
        }
        $fldDataType = $fld->DataType;
        $value = ConvertSearchValue($fldVal, $fldOpr, $fld); // For testing if numeric only
        $value2 = ConvertSearchValue($fldVal2, $fldOpr2, $fld); // For testing if numeric only
        $fldOpr = ConvertSearchOperator($fldOpr, $fld, $value);
        $fldOpr2 = ConvertSearchOperator($fldOpr2, $fld, $value2);
        if (in_array($fldOpr, ["BETWEEN", "NOT BETWEEN"])) {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld) && IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal != "" && $fldVal2 != "" && $isValidValue) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&y_" . $fldParm . "=" . urlencode($fldVal2) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            }
        } else {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld);
            if ($fldVal != "" && $isValidValue && IsValidOperator($fldOpr)) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            } elseif (in_array($fldOpr, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr != "" && $oprOnly && IsValidOperator($fldOpr))) {
                $wrk = "z_" . $fldParm . "=" . urlencode($fldOpr);
            }
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal2 != "" && $isValidValue && IsValidOperator($fldOpr2)) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "y_" . $fldParm . "=" . urlencode($fldVal2) . "&w_" . $fldParm . "=" . urlencode($fldOpr2);
            } elseif (in_array($fldOpr2, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr2 != "" && $oprOnly && IsValidOperator($fldOpr2))) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "w_" . $fldParm . "=" . urlencode($fldOpr2);
            }
        }
        if ($wrk != "") {
            if ($url != "") {
                $url .= "&";
            }
            $url .= $wrk;
        }
    }

    // Load search values for validation
    protected function loadSearchValues()
    {
        // Load search values
        $hasValue = false;

        // id
        if ($this->id->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // status
        if ($this->status->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // buyer_id
        if ($this->buyer_id->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // product_ref
        if ($this->product_ref->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // current_reading
        if ($this->current_reading->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // product_status
        if ($this->product_status->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // error
        if ($this->_error->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // last_serve
        if ($this->last_serve->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // last_serve_by
        if ($this->last_serve_by->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // query_by
        if ($this->query_by->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // service_executive
        if ($this->service_executive->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // Ins_service_executive
        if ($this->Ins_service_executive->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // operator
        if ($this->operator->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // datetime
        if ($this->datetime->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // o_operator
        if ($this->o_operator->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // o_datetime
        if ($this->o_datetime->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // master_reading
        if ($this->master_reading->AdvancedSearch->get()) {
            $hasValue = true;
        }
        return $hasValue;
    }

    // Render row values based on field settings
    public function renderRow()
    {
        global $Security, $Language, $CurrentLanguage;

        // Initialize URLs

        // Call Row_Rendering event
        $this->rowRendering();

        // Common render codes for all row types

        // id
        $this->id->RowCssClass = "row";

        // status
        $this->status->RowCssClass = "row";

        // buyer_id
        $this->buyer_id->RowCssClass = "row";

        // product_ref
        $this->product_ref->RowCssClass = "row";

        // current_reading
        $this->current_reading->RowCssClass = "row";

        // product_status
        $this->product_status->RowCssClass = "row";

        // error
        $this->_error->RowCssClass = "row";

        // last_serve
        $this->last_serve->RowCssClass = "row";

        // last_serve_by
        $this->last_serve_by->RowCssClass = "row";

        // query_by
        $this->query_by->RowCssClass = "row";

        // service_executive
        $this->service_executive->RowCssClass = "row";

        // Ins_service_executive
        $this->Ins_service_executive->RowCssClass = "row";

        // operator
        $this->operator->RowCssClass = "row";

        // datetime
        $this->datetime->RowCssClass = "row";

        // o_operator
        $this->o_operator->RowCssClass = "row";

        // o_datetime
        $this->o_datetime->RowCssClass = "row";

        // rid_1
        $this->rid_1->RowCssClass = "row";

        // master_reading
        $this->master_reading->RowCssClass = "row";

        // View row
        if ($this->RowType == RowType::VIEW) {
            // id
            $this->id->ViewValue = $this->id->CurrentValue;

            // status
            if (strval($this->status->CurrentValue) != "") {
                $this->status->ViewValue = $this->status->optionCaption($this->status->CurrentValue);
            } else {
                $this->status->ViewValue = null;
            }

            // buyer_id
            $this->buyer_id->ViewValue = $this->buyer_id->CurrentValue;
            $curVal = strval($this->buyer_id->CurrentValue);
            if ($curVal != "") {
                $this->buyer_id->ViewValue = $this->buyer_id->lookupCacheOption($curVal);
                if ($this->buyer_id->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->buyer_id->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->buyer_id->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                    $lookupFilter = $this->buyer_id->getSelectFilter($this); // PHP
                    $sqlWrk = $this->buyer_id->Lookup->getSql(false, $filterWrk, $lookupFilter, $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->buyer_id->Lookup->renderViewRow($rswrk[0]);
                        $this->buyer_id->ViewValue = $this->buyer_id->displayValue($arwrk);
                    } else {
                        $this->buyer_id->ViewValue = FormatNumber($this->buyer_id->CurrentValue, $this->buyer_id->formatPattern());
                    }
                }
            } else {
                $this->buyer_id->ViewValue = null;
            }

            // product_ref
            $curVal = strval($this->product_ref->CurrentValue);
            if ($curVal != "") {
                $this->product_ref->ViewValue = $this->product_ref->lookupCacheOption($curVal);
                if ($this->product_ref->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->product_ref->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->product_ref->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                    $sqlWrk = $this->product_ref->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->product_ref->Lookup->renderViewRow($rswrk[0]);
                        $this->product_ref->ViewValue = $this->product_ref->displayValue($arwrk);
                    } else {
                        $this->product_ref->ViewValue = FormatNumber($this->product_ref->CurrentValue, $this->product_ref->formatPattern());
                    }
                }
            } else {
                $this->product_ref->ViewValue = null;
            }

            // current_reading
            $this->current_reading->ViewValue = $this->current_reading->CurrentValue;
            $this->current_reading->ViewValue = FormatNumber($this->current_reading->ViewValue, $this->current_reading->formatPattern());

            // product_status
            $curVal = strval($this->product_status->CurrentValue);
            if ($curVal != "") {
                $this->product_status->ViewValue = $this->product_status->lookupCacheOption($curVal);
                if ($this->product_status->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->product_status->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->product_status->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                    $lookupFilter = $this->product_status->getSelectFilter($this); // PHP
                    $sqlWrk = $this->product_status->Lookup->getSql(false, $filterWrk, $lookupFilter, $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->product_status->Lookup->renderViewRow($rswrk[0]);
                        $this->product_status->ViewValue = $this->product_status->displayValue($arwrk);
                    } else {
                        $this->product_status->ViewValue = $this->product_status->CurrentValue;
                    }
                }
            } else {
                $this->product_status->ViewValue = null;
            }

            // error
            $this->_error->ViewValue = $this->_error->CurrentValue;

            // last_serve
            $this->last_serve->ViewValue = $this->last_serve->CurrentValue;
            $this->last_serve->ViewValue = FormatDateTime($this->last_serve->ViewValue, $this->last_serve->formatPattern());

            // last_serve_by
            $curVal = strval($this->last_serve_by->CurrentValue);
            if ($curVal != "") {
                $this->last_serve_by->ViewValue = $this->last_serve_by->lookupCacheOption($curVal);
                if ($this->last_serve_by->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->last_serve_by->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->last_serve_by->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                    $lookupFilter = $this->last_serve_by->getSelectFilter($this); // PHP
                    $sqlWrk = $this->last_serve_by->Lookup->getSql(false, $filterWrk, $lookupFilter, $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->last_serve_by->Lookup->renderViewRow($rswrk[0]);
                        $this->last_serve_by->ViewValue = $this->last_serve_by->displayValue($arwrk);
                    } else {
                        $this->last_serve_by->ViewValue = FormatNumber($this->last_serve_by->CurrentValue, $this->last_serve_by->formatPattern());
                    }
                }
            } else {
                $this->last_serve_by->ViewValue = null;
            }

            // query_by
            $this->query_by->ViewValue = $this->query_by->CurrentValue;

            // service_executive
            $curVal = strval($this->service_executive->CurrentValue);
            if ($curVal != "") {
                $this->service_executive->ViewValue = $this->service_executive->lookupCacheOption($curVal);
                if ($this->service_executive->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->service_executive->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->service_executive->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                    $lookupFilter = $this->service_executive->getSelectFilter($this); // PHP
                    $sqlWrk = $this->service_executive->Lookup->getSql(false, $filterWrk, $lookupFilter, $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->service_executive->Lookup->renderViewRow($rswrk[0]);
                        $this->service_executive->ViewValue = $this->service_executive->displayValue($arwrk);
                    } else {
                        $this->service_executive->ViewValue = FormatNumber($this->service_executive->CurrentValue, $this->service_executive->formatPattern());
                    }
                }
            } else {
                $this->service_executive->ViewValue = null;
            }

            // Ins_service_executive
            $this->Ins_service_executive->ViewValue = $this->Ins_service_executive->CurrentValue;

            // operator
            $this->operator->ViewValue = $this->operator->CurrentValue;
            $this->operator->ViewValue = FormatNumber($this->operator->ViewValue, $this->operator->formatPattern());

            // datetime
            $this->datetime->ViewValue = $this->datetime->CurrentValue;
            $this->datetime->ViewValue = FormatDateTime($this->datetime->ViewValue, $this->datetime->formatPattern());

            // o_operator
            $this->o_operator->ViewValue = $this->o_operator->CurrentValue;
            $this->o_operator->ViewValue = FormatNumber($this->o_operator->ViewValue, $this->o_operator->formatPattern());

            // o_datetime
            $this->o_datetime->ViewValue = $this->o_datetime->CurrentValue;
            $this->o_datetime->ViewValue = FormatDateTime($this->o_datetime->ViewValue, $this->o_datetime->formatPattern());

            // master_reading
            $this->master_reading->ViewValue = $this->master_reading->CurrentValue;
            $this->master_reading->ViewValue = FormatNumber($this->master_reading->ViewValue, $this->master_reading->formatPattern());

            // id
            $this->id->HrefValue = "";
            $this->id->TooltipValue = "";

            // status
            $this->status->HrefValue = "";
            $this->status->TooltipValue = "";

            // buyer_id
            $this->buyer_id->HrefValue = "";
            $this->buyer_id->TooltipValue = "";

            // product_ref
            $this->product_ref->HrefValue = "";
            $this->product_ref->TooltipValue = "";

            // current_reading
            $this->current_reading->HrefValue = "";
            $this->current_reading->TooltipValue = "";

            // product_status
            $this->product_status->HrefValue = "";
            $this->product_status->TooltipValue = "";

            // error
            $this->_error->HrefValue = "";
            $this->_error->TooltipValue = "";

            // last_serve
            $this->last_serve->HrefValue = "";
            $this->last_serve->TooltipValue = "";

            // last_serve_by
            $this->last_serve_by->HrefValue = "";
            $this->last_serve_by->TooltipValue = "";

            // query_by
            $this->query_by->HrefValue = "";
            $this->query_by->TooltipValue = "";

            // service_executive
            $this->service_executive->HrefValue = "";
            $this->service_executive->TooltipValue = "";

            // Ins_service_executive
            $this->Ins_service_executive->HrefValue = "";
            $this->Ins_service_executive->TooltipValue = "";

            // operator
            $this->operator->HrefValue = "";
            $this->operator->TooltipValue = "";

            // datetime
            $this->datetime->HrefValue = "";
            $this->datetime->TooltipValue = "";

            // o_operator
            $this->o_operator->HrefValue = "";
            $this->o_operator->TooltipValue = "";

            // o_datetime
            $this->o_datetime->HrefValue = "";
            $this->o_datetime->TooltipValue = "";

            // master_reading
            $this->master_reading->HrefValue = "";
            $this->master_reading->TooltipValue = "";
        } elseif ($this->RowType == RowType::SEARCH) {
            // id
            $this->id->setupEditAttributes();
            $this->id->EditValue = $this->id->AdvancedSearch->SearchValue;
            $this->id->PlaceHolder = RemoveHtml($this->id->caption());

            // status
            $this->status->setupEditAttributes();
            $this->status->EditValue = $this->status->options(true);
            $this->status->PlaceHolder = RemoveHtml($this->status->caption());

            // buyer_id
            $this->buyer_id->setupEditAttributes();
            $this->buyer_id->EditValue = $this->buyer_id->AdvancedSearch->SearchValue;
            $curVal = strval($this->buyer_id->AdvancedSearch->SearchValue);
            if ($curVal != "") {
                $this->buyer_id->EditValue = $this->buyer_id->lookupCacheOption($curVal);
                if ($this->buyer_id->EditValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->buyer_id->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->buyer_id->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                    $lookupFilter = $this->buyer_id->getSelectFilter($this); // PHP
                    $sqlWrk = $this->buyer_id->Lookup->getSql(false, $filterWrk, $lookupFilter, $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->buyer_id->Lookup->renderViewRow($rswrk[0]);
                        $this->buyer_id->EditValue = $this->buyer_id->displayValue($arwrk);
                    } else {
                        $this->buyer_id->EditValue = HtmlEncode(FormatNumber($this->buyer_id->AdvancedSearch->SearchValue, $this->buyer_id->formatPattern()));
                    }
                }
            } else {
                $this->buyer_id->EditValue = null;
            }
            $this->buyer_id->PlaceHolder = RemoveHtml($this->buyer_id->caption());

            // product_ref
            $curVal = trim(strval($this->product_ref->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->product_ref->AdvancedSearch->ViewValue = $this->product_ref->lookupCacheOption($curVal);
            } else {
                $this->product_ref->AdvancedSearch->ViewValue = $this->product_ref->Lookup !== null && is_array($this->product_ref->lookupOptions()) && count($this->product_ref->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->product_ref->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->product_ref->EditValue = array_values($this->product_ref->lookupOptions());
                if ($this->product_ref->AdvancedSearch->ViewValue == "") {
                    $this->product_ref->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->product_ref->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $this->product_ref->AdvancedSearch->SearchValue, $this->product_ref->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                }
                $sqlWrk = $this->product_ref->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                if ($ari > 0) { // Lookup values found
                    $arwrk = $this->product_ref->Lookup->renderViewRow($rswrk[0]);
                    $this->product_ref->AdvancedSearch->ViewValue = $this->product_ref->displayValue($arwrk);
                } else {
                    $this->product_ref->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
                $arwrk = $rswrk;
                $this->product_ref->EditValue = $arwrk;
            }
            $this->product_ref->PlaceHolder = RemoveHtml($this->product_ref->caption());

            // current_reading
            $this->current_reading->setupEditAttributes();
            $this->current_reading->EditValue = $this->current_reading->AdvancedSearch->SearchValue;
            $this->current_reading->PlaceHolder = RemoveHtml($this->current_reading->caption());

            // product_status
            $this->product_status->setupEditAttributes();
            $curVal = trim(strval($this->product_status->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->product_status->AdvancedSearch->ViewValue = $this->product_status->lookupCacheOption($curVal);
            } else {
                $this->product_status->AdvancedSearch->ViewValue = $this->product_status->Lookup !== null && is_array($this->product_status->lookupOptions()) && count($this->product_status->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->product_status->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->product_status->EditValue = array_values($this->product_status->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->product_status->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $this->product_status->AdvancedSearch->SearchValue, $this->product_status->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                }
                $lookupFilter = $this->product_status->getSelectFilter($this); // PHP
                $sqlWrk = $this->product_status->Lookup->getSql(true, $filterWrk, $lookupFilter, $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->product_status->EditValue = $arwrk;
            }
            $this->product_status->PlaceHolder = RemoveHtml($this->product_status->caption());

            // error
            $this->_error->setupEditAttributes();
            $this->_error->EditValue = HtmlEncode($this->_error->AdvancedSearch->SearchValue);
            $this->_error->PlaceHolder = RemoveHtml($this->_error->caption());

            // last_serve
            $this->last_serve->setupEditAttributes();
            $this->last_serve->EditValue = HtmlEncode(FormatDateTime(UnFormatDateTime($this->last_serve->AdvancedSearch->SearchValue, $this->last_serve->formatPattern()), $this->last_serve->formatPattern()));
            $this->last_serve->PlaceHolder = RemoveHtml($this->last_serve->caption());

            // last_serve_by
            $this->last_serve_by->setupEditAttributes();
            $curVal = trim(strval($this->last_serve_by->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->last_serve_by->AdvancedSearch->ViewValue = $this->last_serve_by->lookupCacheOption($curVal);
            } else {
                $this->last_serve_by->AdvancedSearch->ViewValue = $this->last_serve_by->Lookup !== null && is_array($this->last_serve_by->lookupOptions()) && count($this->last_serve_by->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->last_serve_by->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->last_serve_by->EditValue = array_values($this->last_serve_by->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->last_serve_by->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $this->last_serve_by->AdvancedSearch->SearchValue, $this->last_serve_by->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                }
                $lookupFilter = $this->last_serve_by->getSelectFilter($this); // PHP
                $sqlWrk = $this->last_serve_by->Lookup->getSql(true, $filterWrk, $lookupFilter, $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->last_serve_by->EditValue = $arwrk;
            }
            $this->last_serve_by->PlaceHolder = RemoveHtml($this->last_serve_by->caption());

            // query_by
            $this->query_by->setupEditAttributes();
            if (!$this->query_by->Raw) {
                $this->query_by->AdvancedSearch->SearchValue = HtmlDecode($this->query_by->AdvancedSearch->SearchValue);
            }
            $this->query_by->EditValue = HtmlEncode($this->query_by->AdvancedSearch->SearchValue);
            $this->query_by->PlaceHolder = RemoveHtml($this->query_by->caption());

            // service_executive
            $this->service_executive->setupEditAttributes();
            $curVal = trim(strval($this->service_executive->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->service_executive->AdvancedSearch->ViewValue = $this->service_executive->lookupCacheOption($curVal);
            } else {
                $this->service_executive->AdvancedSearch->ViewValue = $this->service_executive->Lookup !== null && is_array($this->service_executive->lookupOptions()) && count($this->service_executive->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->service_executive->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->service_executive->EditValue = array_values($this->service_executive->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->service_executive->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $this->service_executive->AdvancedSearch->SearchValue, $this->service_executive->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                }
                $lookupFilter = $this->service_executive->getSelectFilter($this); // PHP
                $sqlWrk = $this->service_executive->Lookup->getSql(true, $filterWrk, $lookupFilter, $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->service_executive->EditValue = $arwrk;
            }
            $this->service_executive->PlaceHolder = RemoveHtml($this->service_executive->caption());

            // Ins_service_executive
            $this->Ins_service_executive->setupEditAttributes();
            $this->Ins_service_executive->EditValue = HtmlEncode($this->Ins_service_executive->AdvancedSearch->SearchValue);
            $this->Ins_service_executive->PlaceHolder = RemoveHtml($this->Ins_service_executive->caption());

            // operator
            $this->operator->setupEditAttributes();
            $this->operator->EditValue = $this->operator->AdvancedSearch->SearchValue;
            $this->operator->PlaceHolder = RemoveHtml($this->operator->caption());

            // datetime
            $this->datetime->setupEditAttributes();
            $this->datetime->EditValue = HtmlEncode(FormatDateTime(UnFormatDateTime($this->datetime->AdvancedSearch->SearchValue, $this->datetime->formatPattern()), $this->datetime->formatPattern()));
            $this->datetime->PlaceHolder = RemoveHtml($this->datetime->caption());

            // o_operator
            $this->o_operator->setupEditAttributes();
            $this->o_operator->EditValue = $this->o_operator->AdvancedSearch->SearchValue;
            $this->o_operator->PlaceHolder = RemoveHtml($this->o_operator->caption());

            // o_datetime
            $this->o_datetime->setupEditAttributes();
            $this->o_datetime->EditValue = HtmlEncode(FormatDateTime(UnFormatDateTime($this->o_datetime->AdvancedSearch->SearchValue, $this->o_datetime->formatPattern()), $this->o_datetime->formatPattern()));
            $this->o_datetime->PlaceHolder = RemoveHtml($this->o_datetime->caption());

            // master_reading
            $this->master_reading->setupEditAttributes();
            $this->master_reading->EditValue = $this->master_reading->AdvancedSearch->SearchValue;
            $this->master_reading->PlaceHolder = RemoveHtml($this->master_reading->caption());
        }
        if ($this->RowType == RowType::ADD || $this->RowType == RowType::EDIT || $this->RowType == RowType::SEARCH) { // Add/Edit/Search row
            $this->setupFieldTitles();
        }

        // Call Row Rendered event
        if ($this->RowType != RowType::AGGREGATEINIT) {
            $this->rowRendered();
        }
    }

    // Validate search
    protected function validateSearch()
    {
        // Check if validation required
        if (!Config("SERVER_VALIDATE")) {
            return true;
        }
        if (!CheckInteger($this->id->AdvancedSearch->SearchValue)) {
            $this->id->addErrorMessage($this->id->getErrorMessage(false));
        }
        if (!CheckInteger($this->buyer_id->AdvancedSearch->SearchValue)) {
            $this->buyer_id->addErrorMessage($this->buyer_id->getErrorMessage(false));
        }
        if (!CheckInteger($this->current_reading->AdvancedSearch->SearchValue)) {
            $this->current_reading->addErrorMessage($this->current_reading->getErrorMessage(false));
        }
        if (!CheckDate($this->last_serve->AdvancedSearch->SearchValue, $this->last_serve->formatPattern())) {
            $this->last_serve->addErrorMessage($this->last_serve->getErrorMessage(false));
        }
        if (!CheckInteger($this->operator->AdvancedSearch->SearchValue)) {
            $this->operator->addErrorMessage($this->operator->getErrorMessage(false));
        }
        if (!CheckDate($this->datetime->AdvancedSearch->SearchValue, $this->datetime->formatPattern())) {
            $this->datetime->addErrorMessage($this->datetime->getErrorMessage(false));
        }
        if (!CheckInteger($this->o_operator->AdvancedSearch->SearchValue)) {
            $this->o_operator->addErrorMessage($this->o_operator->getErrorMessage(false));
        }
        if (!CheckDate($this->o_datetime->AdvancedSearch->SearchValue, $this->o_datetime->formatPattern())) {
            $this->o_datetime->addErrorMessage($this->o_datetime->getErrorMessage(false));
        }
        if (!CheckInteger($this->master_reading->AdvancedSearch->SearchValue)) {
            $this->master_reading->addErrorMessage($this->master_reading->getErrorMessage(false));
        }

        // Return validate result
        $validateSearch = !$this->hasInvalidFields();

        // Call Form_CustomValidate event
        $formCustomError = "";
        $validateSearch = $validateSearch && $this->formCustomValidate($formCustomError);
        if ($formCustomError != "") {
            $this->setFailureMessage($formCustomError);
        }
        return $validateSearch;
    }

    // Load advanced search
    public function loadAdvancedSearch()
    {
        $this->id->AdvancedSearch->load();
        $this->status->AdvancedSearch->load();
        $this->buyer_id->AdvancedSearch->load();
        $this->product_ref->AdvancedSearch->load();
        $this->current_reading->AdvancedSearch->load();
        $this->product_status->AdvancedSearch->load();
        $this->_error->AdvancedSearch->load();
        $this->last_serve->AdvancedSearch->load();
        $this->last_serve_by->AdvancedSearch->load();
        $this->query_by->AdvancedSearch->load();
        $this->service_executive->AdvancedSearch->load();
        $this->Ins_service_executive->AdvancedSearch->load();
        $this->operator->AdvancedSearch->load();
        $this->datetime->AdvancedSearch->load();
        $this->o_operator->AdvancedSearch->load();
        $this->o_datetime->AdvancedSearch->load();
        $this->master_reading->AdvancedSearch->load();
    }

    // Set up Breadcrumb
    protected function setupBreadcrumb()
    {
        global $Breadcrumb, $Language;
        $Breadcrumb = new Breadcrumb("db2");
        $url = CurrentUrl();
        $Breadcrumb->add("list", $this->TableVar, $this->addMasterUrl("servicerequestlist"), "", $this->TableVar, true);
        $pageId = "search";
        $Breadcrumb->add("search", $pageId, $url);
    }

    // Setup lookup options
    public function setupLookupOptions($fld)
    {
        if ($fld->Lookup && $fld->Lookup->Options === null) {
            // Get default connection and filter
            $conn = $this->getConnection();
            $lookupFilter = "";

            // No need to check any more
            $fld->Lookup->Options = [];

            // Set up lookup SQL and connection
            switch ($fld->FieldVar) {
                case "x_status":
                    break;
                case "x_buyer_id":
                    $lookupFilter = $fld->getSelectFilter(); // PHP
                    break;
                case "x_product_ref":
                    break;
                case "x_product_status":
                    $lookupFilter = $fld->getSelectFilter(); // PHP
                    break;
                case "x_last_serve_by":
                    $lookupFilter = $fld->getSelectFilter(); // PHP
                    break;
                case "x_service_executive":
                    $lookupFilter = $fld->getSelectFilter(); // PHP
                    break;
                default:
                    $lookupFilter = "";
                    break;
            }

            // Always call to Lookup->getSql so that user can setup Lookup->Options in Lookup_Selecting server event
            $sql = $fld->Lookup->getSql(false, "", $lookupFilter, $this);

            // Set up lookup cache
            if (!$fld->hasLookupOptions() && $fld->UseLookupCache && $sql != "" && count($fld->Lookup->Options) == 0 && count($fld->Lookup->FilterFields) == 0) {
                $totalCnt = $this->getRecordCount($sql, $conn);
                if ($totalCnt > $fld->LookupCacheCount) { // Total count > cache count, do not cache
                    return;
                }
                $rows = $conn->executeQuery($sql)->fetchAll();
                $ar = [];
                foreach ($rows as $row) {
                    $row = $fld->Lookup->renderViewRow($row, Container($fld->Lookup->LinkTable));
                    $key = $row["lf"];
                    if (IsFloatType($fld->Type)) { // Handle float field
                        $key = (float)$key;
                    }
                    $ar[strval($key)] = $row;
                }
                $fld->Lookup->Options = $ar;
            }
        }
    }

    // Page Load event
    public function pageLoad()
    {
        //Log("Page Load");
    }

    // Page Unload event
    public function pageUnload()
    {
        //Log("Page Unload");
    }

    // Page Redirecting event
    public function pageRedirecting(&$url)
    {
        // Example:
        //$url = "your URL";
    }

    // Message Showing event
    // $type = ''|'success'|'failure'|'warning'
    public function messageShowing(&$msg, $type)
    {
        if ($type == "success") {
            //$msg = "your success message";
        } elseif ($type == "failure") {
            //$msg = "your failure message";
        } elseif ($type == "warning") {
            //$msg = "your warning message";
        } else {
            //$msg = "your message";
        }
    }

    // Page Render event
    public function pageRender()
    {
        //Log("Page Render");
    }

    // Page Data Rendering event
    public function pageDataRendering(&$header)
    {
        // Example:
        //$header = "your header";
    }

    // Page Data Rendered event
    public function pageDataRendered(&$footer)
    {
        // Example:
        //$footer = "your footer";
    }

    // Page Breaking event
    public function pageBreaking(&$break, &$content)
    {
        // Example:
        //$break = false; // Skip page break, or
        //$content = "<div style=\"break-after:page;\"></div>"; // Modify page break content
    }

    // Form Custom Validate event
    public function formCustomValidate(&$customError)
    {
        // Return error message in $customError
        return true;
    }
}
