<?php

namespace PHPMaker2024\fibre;

use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use Psr\Http\Message\ServerRequestInterface as Request;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Container\ContainerInterface;
use Slim\Routing\RouteCollectorProxy;
use Slim\App;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Closure;

/**
 * Page class
 */
class PortalVersionSearch extends PortalVersion
{
    use MessagesTrait;

    // Page ID
    public $PageID = "search";

    // Project ID
    public $ProjectID = PROJECT_ID;

    // Page object name
    public $PageObjName = "PortalVersionSearch";

    // View file path
    public $View = null;

    // Title
    public $Title = null; // Title for <title> tag

    // Rendering View
    public $RenderingView = false;

    // CSS class/style
    public $CurrentPageName = "portalversionsearch";

    // Page headings
    public $Heading = "";
    public $Subheading = "";
    public $PageHeader;
    public $PageFooter;

    // Page layout
    public $UseLayout = true;

    // Page terminated
    private $terminated = false;

    // Page heading
    public function pageHeading()
    {
        global $Language;
        if ($this->Heading != "") {
            return $this->Heading;
        }
        if (method_exists($this, "tableCaption")) {
            return $this->tableCaption();
        }
        return "";
    }

    // Page subheading
    public function pageSubheading()
    {
        global $Language;
        if ($this->Subheading != "") {
            return $this->Subheading;
        }
        if ($this->TableName) {
            return $Language->phrase($this->PageID);
        }
        return "";
    }

    // Page name
    public function pageName()
    {
        return CurrentPageName();
    }

    // Page URL
    public function pageUrl($withArgs = true)
    {
        $route = GetRoute();
        $args = RemoveXss($route->getArguments());
        if (!$withArgs) {
            foreach ($args as $key => &$val) {
                $val = "";
            }
            unset($val);
        }
        return rtrim(UrlFor($route->getName(), $args), "/") . "?";
    }

    // Show Page Header
    public function showPageHeader()
    {
        $header = $this->PageHeader;
        $this->pageDataRendering($header);
        if ($header != "") { // Header exists, display
            echo '<div id="ew-page-header">' . $header . '</div>';
        }
    }

    // Show Page Footer
    public function showPageFooter()
    {
        $footer = $this->PageFooter;
        $this->pageDataRendered($footer);
        if ($footer != "") { // Footer exists, display
            echo '<div id="ew-page-footer">' . $footer . '</div>';
        }
    }

    // Set field visibility
    public function setVisibility()
    {
        $this->id->setVisibility();
        $this->datetime->setVisibility();
        $this->version->setVisibility();
        $this->php_code->setVisibility();
        $this->portal_d_php_code->setVisibility();
        $this->portal_d_apc_query->setVisibility();
        $this->portal_d_mysql_query->setVisibility();
        $this->portal_d_sp_query->setVisibility();
        $this->portal_d_url_link->setVisibility();
        $this->portal_c_php_code->setVisibility();
        $this->mysql_query->setVisibility();
        $this->portal_c_mysql_query->setVisibility();
        $this->sp_query->setVisibility();
        $this->portal_c_sp_query->setVisibility();
        $this->apc_query->setVisibility();
        $this->portal_c_apc_query->setVisibility();
        $this->url_link->setVisibility();
        $this->u_type->setVisibility();
    }

    // Constructor
    public function __construct()
    {
        parent::__construct();
        global $Language, $DashboardReport, $DebugTimer, $UserTable;
        $this->TableVar = 'portal_version';
        $this->TableName = 'portal_version';

        // Table CSS class
        $this->TableClass = "table table-striped table-bordered table-hover table-sm ew-desktop-table ew-search-table";

        // Initialize
        $GLOBALS["Page"] = &$this;

        // Language object
        $Language = Container("app.language");

        // Table object (portal_version)
        if (!isset($GLOBALS["portal_version"]) || $GLOBALS["portal_version"]::class == PROJECT_NAMESPACE . "portal_version") {
            $GLOBALS["portal_version"] = &$this;
        }

        // Table name (for backward compatibility only)
        if (!defined(PROJECT_NAMESPACE . "TABLE_NAME")) {
            define(PROJECT_NAMESPACE . "TABLE_NAME", 'portal_version');
        }

        // Start timer
        $DebugTimer = Container("debug.timer");

        // Debug message
        LoadDebugMessage();

        // Open connection
        $GLOBALS["Conn"] ??= $this->getConnection();

        // User table object
        $UserTable = Container("usertable");
    }

    // Get content from stream
    public function getContents(): string
    {
        global $Response;
        return $Response?->getBody() ?? ob_get_clean();
    }

    // Is lookup
    public function isLookup()
    {
        return SameText(Route(0), Config("API_LOOKUP_ACTION"));
    }

    // Is AutoFill
    public function isAutoFill()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autofill");
    }

    // Is AutoSuggest
    public function isAutoSuggest()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autosuggest");
    }

    // Is modal lookup
    public function isModalLookup()
    {
        return $this->isLookup() && SameText(Post("ajax"), "modal");
    }

    // Is terminated
    public function isTerminated()
    {
        return $this->terminated;
    }

    /**
     * Terminate page
     *
     * @param string $url URL for direction
     * @return void
     */
    public function terminate($url = "")
    {
        if ($this->terminated) {
            return;
        }
        global $TempImages, $DashboardReport, $Response;

        // Page is terminated
        $this->terminated = true;

        // Page Unload event
        if (method_exists($this, "pageUnload")) {
            $this->pageUnload();
        }
        DispatchEvent(new PageUnloadedEvent($this), PageUnloadedEvent::NAME);
        if (!IsApi() && method_exists($this, "pageRedirecting")) {
            $this->pageRedirecting($url);
        }

        // Close connection
        CloseConnections();

        // Return for API
        if (IsApi()) {
            $res = $url === true;
            if (!$res) { // Show response for API
                $ar = array_merge($this->getMessages(), $url ? ["url" => GetUrl($url)] : []);
                WriteJson($ar);
            }
            $this->clearMessages(); // Clear messages for API request
            return;
        } else { // Check if response is JSON
            if (WithJsonResponse()) { // With JSON response
                $this->clearMessages();
                return;
            }
        }

        // Go to URL if specified
        if ($url != "") {
            if (!Config("DEBUG") && ob_get_length()) {
                ob_end_clean();
            }

            // Handle modal response
            if ($this->IsModal) { // Show as modal
                $pageName = GetPageName($url);
                $result = ["url" => GetUrl($url), "modal" => "1"];  // Assume return to modal for simplicity
                if (!SameString($pageName, GetPageName($this->getListUrl()))) { // Not List page
                    $result["caption"] = $this->getModalCaption($pageName);
                    $result["view"] = SameString($pageName, "portalversionview"); // If View page, no primary button
                } else { // List page
                    $result["error"] = $this->getFailureMessage(); // List page should not be shown as modal => error
                    $this->clearFailureMessage();
                }
                WriteJson($result);
            } else {
                SaveDebugMessage();
                Redirect(GetUrl($url));
            }
        }
        return; // Return to controller
    }

    // Get records from result set
    protected function getRecordsFromRecordset($rs, $current = false)
    {
        $rows = [];
        if (is_object($rs)) { // Result set
            while ($row = $rs->fetch()) {
                $this->loadRowValues($row); // Set up DbValue/CurrentValue
                $row = $this->getRecordFromArray($row);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        } elseif (is_array($rs)) {
            foreach ($rs as $ar) {
                $row = $this->getRecordFromArray($ar);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        }
        return $rows;
    }

    // Get record from array
    protected function getRecordFromArray($ar)
    {
        $row = [];
        if (is_array($ar)) {
            foreach ($ar as $fldname => $val) {
                if (array_key_exists($fldname, $this->Fields) && ($this->Fields[$fldname]->Visible || $this->Fields[$fldname]->IsPrimaryKey)) { // Primary key or Visible
                    $fld = &$this->Fields[$fldname];
                    if ($fld->HtmlTag == "FILE") { // Upload field
                        if (EmptyValue($val)) {
                            $row[$fldname] = null;
                        } else {
                            if ($fld->DataType == DataType::BLOB) {
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . $fld->Param . "/" . rawurlencode($this->getRecordKeyValue($ar))));
                                $row[$fldname] = ["type" => ContentType($val), "url" => $url, "name" => $fld->Param . ContentExtension($val)];
                            } elseif (!$fld->UploadMultiple || !ContainsString($val, Config("MULTIPLE_UPLOAD_SEPARATOR"))) { // Single file
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $val)));
                                $row[$fldname] = ["type" => MimeContentType($val), "url" => $url, "name" => $val];
                            } else { // Multiple files
                                $files = explode(Config("MULTIPLE_UPLOAD_SEPARATOR"), $val);
                                $ar = [];
                                foreach ($files as $file) {
                                    $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                        "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $file)));
                                    if (!EmptyValue($file)) {
                                        $ar[] = ["type" => MimeContentType($file), "url" => $url, "name" => $file];
                                    }
                                }
                                $row[$fldname] = $ar;
                            }
                        }
                    } else {
                        $row[$fldname] = $val;
                    }
                }
            }
        }
        return $row;
    }

    // Get record key value from array
    protected function getRecordKeyValue($ar)
    {
        $key = "";
        if (is_array($ar)) {
            $key .= @$ar['id'];
        }
        return $key;
    }

    /**
     * Hide fields for add/edit
     *
     * @return void
     */
    protected function hideFieldsForAddEdit()
    {
        if ($this->isAdd() || $this->isCopy() || $this->isGridAdd()) {
            $this->id->Visible = false;
        }
    }

    // Lookup data
    public function lookup(array $req = [], bool $response = true)
    {
        global $Language, $Security;

        // Get lookup object
        $fieldName = $req["field"] ?? null;
        if (!$fieldName) {
            return [];
        }
        $fld = $this->Fields[$fieldName];
        $lookup = $fld->Lookup;
        $name = $req["name"] ?? "";
        if (ContainsString($name, "query_builder_rule")) {
            $lookup->FilterFields = []; // Skip parent fields if any
        }

        // Get lookup parameters
        $lookupType = $req["ajax"] ?? "unknown";
        $pageSize = -1;
        $offset = -1;
        $searchValue = "";
        if (SameText($lookupType, "modal") || SameText($lookupType, "filter")) {
            $searchValue = $req["q"] ?? $req["sv"] ?? "";
            $pageSize = $req["n"] ?? $req["recperpage"] ?? 10;
        } elseif (SameText($lookupType, "autosuggest")) {
            $searchValue = $req["q"] ?? "";
            $pageSize = $req["n"] ?? -1;
            $pageSize = is_numeric($pageSize) ? (int)$pageSize : -1;
            if ($pageSize <= 0) {
                $pageSize = Config("AUTO_SUGGEST_MAX_ENTRIES");
            }
        }
        $start = $req["start"] ?? -1;
        $start = is_numeric($start) ? (int)$start : -1;
        $page = $req["page"] ?? -1;
        $page = is_numeric($page) ? (int)$page : -1;
        $offset = $start >= 0 ? $start : ($page > 0 && $pageSize > 0 ? ($page - 1) * $pageSize : 0);
        $userSelect = Decrypt($req["s"] ?? "");
        $userFilter = Decrypt($req["f"] ?? "");
        $userOrderBy = Decrypt($req["o"] ?? "");
        $keys = $req["keys"] ?? null;
        $lookup->LookupType = $lookupType; // Lookup type
        $lookup->FilterValues = []; // Clear filter values first
        if ($keys !== null) { // Selected records from modal
            if (is_array($keys)) {
                $keys = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $keys);
            }
            $lookup->FilterFields = []; // Skip parent fields if any
            $lookup->FilterValues[] = $keys; // Lookup values
            $pageSize = -1; // Show all records
        } else { // Lookup values
            $lookup->FilterValues[] = $req["v0"] ?? $req["lookupValue"] ?? "";
        }
        $cnt = is_array($lookup->FilterFields) ? count($lookup->FilterFields) : 0;
        for ($i = 1; $i <= $cnt; $i++) {
            $lookup->FilterValues[] = $req["v" . $i] ?? "";
        }
        $lookup->SearchValue = $searchValue;
        $lookup->PageSize = $pageSize;
        $lookup->Offset = $offset;
        if ($userSelect != "") {
            $lookup->UserSelect = $userSelect;
        }
        if ($userFilter != "") {
            $lookup->UserFilter = $userFilter;
        }
        if ($userOrderBy != "") {
            $lookup->UserOrderBy = $userOrderBy;
        }
        return $lookup->toJson($this, $response); // Use settings from current page
    }
    public $FormClassName = "ew-form ew-search-form";
    public $IsModal = false;
    public $IsMobileOrModal = false;
    public $MultiPages; // Multi pages object

    /**
     * Page run
     *
     * @return void
     */
    public function run()
    {
        global $ExportType, $Language, $Security, $CurrentForm, $SkipHeaderFooter;

        // Is modal
        $this->IsModal = ConvertToBool(Param("modal"));
        $this->UseLayout = $this->UseLayout && !$this->IsModal;

        // Use layout
        $this->UseLayout = $this->UseLayout && ConvertToBool(Param(Config("PAGE_LAYOUT"), true));

        // View
        $this->View = Get(Config("VIEW"));

        // Load user profile
        if (IsLoggedIn()) {
            Profile()->setUserName(CurrentUserName())->loadFromStorage();
        }

        // Create form object
        $CurrentForm = new HttpForm();
        $this->CurrentAction = Param("action"); // Set up current action
        $this->setVisibility();

        // Set lookup cache
        if (!in_array($this->PageID, Config("LOOKUP_CACHE_PAGE_IDS"))) {
            $this->setUseLookupCache(false);
        }

        // Set up multi page object
        $this->setupMultiPages();

        // Global Page Loading event (in userfn*.php)
        DispatchEvent(new PageLoadingEvent($this), PageLoadingEvent::NAME);

        // Page Load event
        if (method_exists($this, "pageLoad")) {
            $this->pageLoad();
        }

        // Hide fields for add/edit
        if (!$this->UseAjaxActions) {
            $this->hideFieldsForAddEdit();
        }
        // Use inline delete
        if ($this->UseAjaxActions) {
            $this->InlineDelete = true;
        }

        // Set up lookup cache
        $this->setupLookupOptions($this->portal_c_php_code);
        $this->setupLookupOptions($this->portal_c_mysql_query);
        $this->setupLookupOptions($this->portal_c_sp_query);

        // Set up Breadcrumb
        $this->setupBreadcrumb();

        // Check modal
        if ($this->IsModal) {
            $SkipHeaderFooter = true;
        }
        $this->IsMobileOrModal = IsMobile() || $this->IsModal;

        // Get action
        $this->CurrentAction = Post("action");
        if ($this->isSearch()) {
            // Build search string for advanced search, remove blank field
            $this->loadSearchValues(); // Get search values
            $srchStr = $this->validateSearch() ? $this->buildAdvancedSearch() : "";
            if ($srchStr != "") {
                $srchStr = "portalversionlist" . "?" . $srchStr;
                // Do not return Json for UseAjaxActions
                if ($this->IsModal && $this->UseAjaxActions) {
                    $this->IsModal = false;
                }
                $this->terminate($srchStr); // Go to list page
                return;
            }
        }

        // Restore search settings from Session
        if (!$this->hasInvalidFields()) {
            $this->loadAdvancedSearch();
        }

        // Render row for search
        $this->RowType = RowType::SEARCH;
        $this->resetAttributes();
        $this->renderRow();

        // Set LoginStatus / Page_Rendering / Page_Render
        if (!IsApi() && !$this->isTerminated()) {
            // Setup login status
            SetupLoginStatus();

            // Pass login status to client side
            SetClientVar("login", LoginStatus());

            // Global Page Rendering event (in userfn*.php)
            DispatchEvent(new PageRenderingEvent($this), PageRenderingEvent::NAME);

            // Page Render event
            if (method_exists($this, "pageRender")) {
                $this->pageRender();
            }

            // Render search option
            if (method_exists($this, "renderSearchOptions")) {
                $this->renderSearchOptions();
            }
        }
    }

    // Build advanced search
    protected function buildAdvancedSearch()
    {
        $srchUrl = "";
        $this->buildSearchUrl($srchUrl, $this->id); // id
        $this->buildSearchUrl($srchUrl, $this->datetime); // datetime
        $this->buildSearchUrl($srchUrl, $this->version); // version
        $this->buildSearchUrl($srchUrl, $this->php_code); // php_code
        $this->buildSearchUrl($srchUrl, $this->portal_d_php_code); // portal_d_php_code
        $this->buildSearchUrl($srchUrl, $this->portal_d_apc_query); // portal_d_apc_query
        $this->buildSearchUrl($srchUrl, $this->portal_d_mysql_query); // portal_d_mysql_query
        $this->buildSearchUrl($srchUrl, $this->portal_d_sp_query); // portal_d_sp_query
        $this->buildSearchUrl($srchUrl, $this->portal_d_url_link); // portal_d_url_link
        $this->buildSearchUrl($srchUrl, $this->portal_c_php_code); // portal_c_php_code
        $this->buildSearchUrl($srchUrl, $this->mysql_query); // mysql_query
        $this->buildSearchUrl($srchUrl, $this->portal_c_mysql_query); // portal_c_mysql_query
        $this->buildSearchUrl($srchUrl, $this->sp_query); // sp_query
        $this->buildSearchUrl($srchUrl, $this->portal_c_sp_query); // portal_c_sp_query
        $this->buildSearchUrl($srchUrl, $this->apc_query); // apc_query
        $this->buildSearchUrl($srchUrl, $this->portal_c_apc_query); // portal_c_apc_query
        $this->buildSearchUrl($srchUrl, $this->url_link); // url_link
        $this->buildSearchUrl($srchUrl, $this->u_type); // u_type
        if ($srchUrl != "") {
            $srchUrl .= "&";
        }
        $srchUrl .= "cmd=search";
        return $srchUrl;
    }

    // Build search URL
    protected function buildSearchUrl(&$url, $fld, $oprOnly = false)
    {
        global $CurrentForm;
        $wrk = "";
        $fldParm = $fld->Param;
        [
            "value" => $fldVal,
            "operator" => $fldOpr,
            "condition" => $fldCond,
            "value2" => $fldVal2,
            "operator2" => $fldOpr2
        ] = $CurrentForm->getSearchValues($fldParm);
        if (is_array($fldVal)) {
            $fldVal = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal);
        }
        if (is_array($fldVal2)) {
            $fldVal2 = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal2);
        }
        $fldDataType = $fld->DataType;
        $value = ConvertSearchValue($fldVal, $fldOpr, $fld); // For testing if numeric only
        $value2 = ConvertSearchValue($fldVal2, $fldOpr2, $fld); // For testing if numeric only
        $fldOpr = ConvertSearchOperator($fldOpr, $fld, $value);
        $fldOpr2 = ConvertSearchOperator($fldOpr2, $fld, $value2);
        if (in_array($fldOpr, ["BETWEEN", "NOT BETWEEN"])) {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld) && IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal != "" && $fldVal2 != "" && $isValidValue) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&y_" . $fldParm . "=" . urlencode($fldVal2) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            }
        } else {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld);
            if ($fldVal != "" && $isValidValue && IsValidOperator($fldOpr)) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            } elseif (in_array($fldOpr, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr != "" && $oprOnly && IsValidOperator($fldOpr))) {
                $wrk = "z_" . $fldParm . "=" . urlencode($fldOpr);
            }
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal2 != "" && $isValidValue && IsValidOperator($fldOpr2)) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "y_" . $fldParm . "=" . urlencode($fldVal2) . "&w_" . $fldParm . "=" . urlencode($fldOpr2);
            } elseif (in_array($fldOpr2, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr2 != "" && $oprOnly && IsValidOperator($fldOpr2))) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "w_" . $fldParm . "=" . urlencode($fldOpr2);
            }
        }
        if ($wrk != "") {
            if ($url != "") {
                $url .= "&";
            }
            $url .= $wrk;
        }
    }

    // Load search values for validation
    protected function loadSearchValues()
    {
        // Load search values
        $hasValue = false;

        // id
        if ($this->id->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // datetime
        if ($this->datetime->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // version
        if ($this->version->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // php_code
        if ($this->php_code->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_d_php_code
        if ($this->portal_d_php_code->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_d_apc_query
        if ($this->portal_d_apc_query->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_d_mysql_query
        if ($this->portal_d_mysql_query->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_d_sp_query
        if ($this->portal_d_sp_query->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_d_url_link
        if ($this->portal_d_url_link->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_c_php_code
        if ($this->portal_c_php_code->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // mysql_query
        if ($this->mysql_query->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_c_mysql_query
        if ($this->portal_c_mysql_query->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // sp_query
        if ($this->sp_query->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_c_sp_query
        if ($this->portal_c_sp_query->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // apc_query
        if ($this->apc_query->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_c_apc_query
        if ($this->portal_c_apc_query->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // url_link
        if ($this->url_link->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // u_type
        if ($this->u_type->AdvancedSearch->get()) {
            $hasValue = true;
        }
        return $hasValue;
    }

    // Render row values based on field settings
    public function renderRow()
    {
        global $Security, $Language, $CurrentLanguage;

        // Initialize URLs

        // Call Row_Rendering event
        $this->rowRendering();

        // Common render codes for all row types

        // id
        $this->id->RowCssClass = "row";

        // datetime
        $this->datetime->RowCssClass = "row";

        // version
        $this->version->RowCssClass = "row";

        // php_code
        $this->php_code->RowCssClass = "row";

        // portal_d_php_code
        $this->portal_d_php_code->RowCssClass = "row";

        // portal_d_apc_query
        $this->portal_d_apc_query->RowCssClass = "row";

        // portal_d_mysql_query
        $this->portal_d_mysql_query->RowCssClass = "row";

        // portal_d_sp_query
        $this->portal_d_sp_query->RowCssClass = "row";

        // portal_d_url_link
        $this->portal_d_url_link->RowCssClass = "row";

        // portal_c_php_code
        $this->portal_c_php_code->RowCssClass = "row";

        // mysql_query
        $this->mysql_query->RowCssClass = "row";

        // portal_c_mysql_query
        $this->portal_c_mysql_query->RowCssClass = "row";

        // sp_query
        $this->sp_query->RowCssClass = "row";

        // portal_c_sp_query
        $this->portal_c_sp_query->RowCssClass = "row";

        // apc_query
        $this->apc_query->RowCssClass = "row";

        // portal_c_apc_query
        $this->portal_c_apc_query->RowCssClass = "row";

        // url_link
        $this->url_link->RowCssClass = "row";

        // u_type
        $this->u_type->RowCssClass = "row";

        // View row
        if ($this->RowType == RowType::VIEW) {
            // id
            $this->id->ViewValue = $this->id->CurrentValue;

            // datetime
            $this->datetime->ViewValue = $this->datetime->CurrentValue;
            $this->datetime->ViewValue = FormatDateTime($this->datetime->ViewValue, $this->datetime->formatPattern());

            // version
            $this->version->ViewValue = $this->version->CurrentValue;

            // php_code
            $this->php_code->ViewValue = $this->php_code->CurrentValue;

            // portal_d_php_code
            $this->portal_d_php_code->ViewValue = $this->portal_d_php_code->CurrentValue;

            // portal_d_apc_query
            $this->portal_d_apc_query->ViewValue = $this->portal_d_apc_query->CurrentValue;

            // portal_d_mysql_query
            $this->portal_d_mysql_query->ViewValue = $this->portal_d_mysql_query->CurrentValue;

            // portal_d_sp_query
            $this->portal_d_sp_query->ViewValue = $this->portal_d_sp_query->CurrentValue;

            // portal_d_url_link
            $this->portal_d_url_link->ViewValue = $this->portal_d_url_link->CurrentValue;

            // portal_c_php_code
            $this->portal_c_php_code->ViewValue = $this->portal_c_php_code->CurrentValue;
            $curVal = strval($this->portal_c_php_code->CurrentValue);
            if ($curVal != "") {
                $this->portal_c_php_code->ViewValue = $this->portal_c_php_code->lookupCacheOption($curVal);
                if ($this->portal_c_php_code->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->portal_c_php_code->Lookup->getTable()->Fields["url"]->searchExpression(), "=", $curVal, $this->portal_c_php_code->Lookup->getTable()->Fields["url"]->searchDataType(), "");
                    $sqlWrk = $this->portal_c_php_code->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->portal_c_php_code->Lookup->renderViewRow($rswrk[0]);
                        $this->portal_c_php_code->ViewValue = $this->portal_c_php_code->displayValue($arwrk);
                    } else {
                        $this->portal_c_php_code->ViewValue = $this->portal_c_php_code->CurrentValue;
                    }
                }
            } else {
                $this->portal_c_php_code->ViewValue = null;
            }

            // mysql_query
            $this->mysql_query->ViewValue = $this->mysql_query->CurrentValue;

            // portal_c_mysql_query
            $this->portal_c_mysql_query->ViewValue = $this->portal_c_mysql_query->CurrentValue;
            $curVal = strval($this->portal_c_mysql_query->CurrentValue);
            if ($curVal != "") {
                $this->portal_c_mysql_query->ViewValue = $this->portal_c_mysql_query->lookupCacheOption($curVal);
                if ($this->portal_c_mysql_query->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->portal_c_mysql_query->Lookup->getTable()->Fields["url"]->searchExpression(), "=", $curVal, $this->portal_c_mysql_query->Lookup->getTable()->Fields["url"]->searchDataType(), "");
                    $sqlWrk = $this->portal_c_mysql_query->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->portal_c_mysql_query->Lookup->renderViewRow($rswrk[0]);
                        $this->portal_c_mysql_query->ViewValue = $this->portal_c_mysql_query->displayValue($arwrk);
                    } else {
                        $this->portal_c_mysql_query->ViewValue = $this->portal_c_mysql_query->CurrentValue;
                    }
                }
            } else {
                $this->portal_c_mysql_query->ViewValue = null;
            }

            // sp_query
            $this->sp_query->ViewValue = $this->sp_query->CurrentValue;

            // portal_c_sp_query
            $this->portal_c_sp_query->ViewValue = $this->portal_c_sp_query->CurrentValue;
            $curVal = strval($this->portal_c_sp_query->CurrentValue);
            if ($curVal != "") {
                $this->portal_c_sp_query->ViewValue = $this->portal_c_sp_query->lookupCacheOption($curVal);
                if ($this->portal_c_sp_query->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->portal_c_sp_query->Lookup->getTable()->Fields["url"]->searchExpression(), "=", $curVal, $this->portal_c_sp_query->Lookup->getTable()->Fields["url"]->searchDataType(), "");
                    $sqlWrk = $this->portal_c_sp_query->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->portal_c_sp_query->Lookup->renderViewRow($rswrk[0]);
                        $this->portal_c_sp_query->ViewValue = $this->portal_c_sp_query->displayValue($arwrk);
                    } else {
                        $this->portal_c_sp_query->ViewValue = $this->portal_c_sp_query->CurrentValue;
                    }
                }
            } else {
                $this->portal_c_sp_query->ViewValue = null;
            }

            // apc_query
            $this->apc_query->ViewValue = $this->apc_query->CurrentValue;

            // portal_c_apc_query
            $this->portal_c_apc_query->ViewValue = $this->portal_c_apc_query->CurrentValue;

            // url_link
            $this->url_link->ViewValue = $this->url_link->CurrentValue;

            // u_type
            $this->u_type->ViewValue = $this->u_type->CurrentValue;
            $this->u_type->ViewValue = FormatNumber($this->u_type->ViewValue, $this->u_type->formatPattern());

            // id
            $this->id->HrefValue = "";
            $this->id->TooltipValue = "";

            // datetime
            $this->datetime->HrefValue = "";
            $this->datetime->TooltipValue = "";

            // version
            $this->version->HrefValue = "";
            $this->version->TooltipValue = "";

            // php_code
            $this->php_code->HrefValue = "";
            $this->php_code->TooltipValue = "";

            // portal_d_php_code
            $this->portal_d_php_code->HrefValue = "";
            $this->portal_d_php_code->TooltipValue = "";

            // portal_d_apc_query
            $this->portal_d_apc_query->HrefValue = "";
            $this->portal_d_apc_query->TooltipValue = "";

            // portal_d_mysql_query
            $this->portal_d_mysql_query->HrefValue = "";
            $this->portal_d_mysql_query->TooltipValue = "";

            // portal_d_sp_query
            $this->portal_d_sp_query->HrefValue = "";
            $this->portal_d_sp_query->TooltipValue = "";

            // portal_d_url_link
            $this->portal_d_url_link->HrefValue = "";
            $this->portal_d_url_link->TooltipValue = "";

            // portal_c_php_code
            $this->portal_c_php_code->HrefValue = "";
            $this->portal_c_php_code->TooltipValue = "";

            // mysql_query
            $this->mysql_query->HrefValue = "";
            $this->mysql_query->TooltipValue = "";

            // portal_c_mysql_query
            $this->portal_c_mysql_query->HrefValue = "";
            $this->portal_c_mysql_query->TooltipValue = "";

            // sp_query
            $this->sp_query->HrefValue = "";
            $this->sp_query->TooltipValue = "";

            // portal_c_sp_query
            $this->portal_c_sp_query->HrefValue = "";
            $this->portal_c_sp_query->TooltipValue = "";

            // apc_query
            $this->apc_query->HrefValue = "";
            $this->apc_query->TooltipValue = "";

            // portal_c_apc_query
            $this->portal_c_apc_query->HrefValue = "";
            $this->portal_c_apc_query->TooltipValue = "";

            // url_link
            $this->url_link->HrefValue = "";
            $this->url_link->TooltipValue = "";

            // u_type
            $this->u_type->HrefValue = "";
            $this->u_type->TooltipValue = "";
        } elseif ($this->RowType == RowType::SEARCH) {
            // id
            $this->id->setupEditAttributes();
            $this->id->EditValue = $this->id->AdvancedSearch->SearchValue;
            $this->id->PlaceHolder = RemoveHtml($this->id->caption());

            // datetime
            $this->datetime->setupEditAttributes();
            $this->datetime->EditValue = HtmlEncode(FormatDateTime(UnFormatDateTime($this->datetime->AdvancedSearch->SearchValue, $this->datetime->formatPattern()), $this->datetime->formatPattern()));
            $this->datetime->PlaceHolder = RemoveHtml($this->datetime->caption());

            // version
            $this->version->setupEditAttributes();
            if (!$this->version->Raw) {
                $this->version->AdvancedSearch->SearchValue = HtmlDecode($this->version->AdvancedSearch->SearchValue);
            }
            $this->version->EditValue = HtmlEncode($this->version->AdvancedSearch->SearchValue);
            $this->version->PlaceHolder = RemoveHtml($this->version->caption());

            // php_code
            $this->php_code->setupEditAttributes();
            if (!$this->php_code->Raw) {
                $this->php_code->AdvancedSearch->SearchValue = HtmlDecode($this->php_code->AdvancedSearch->SearchValue);
            }
            $this->php_code->EditValue = HtmlEncode($this->php_code->AdvancedSearch->SearchValue);
            $this->php_code->PlaceHolder = RemoveHtml($this->php_code->caption());

            // portal_d_php_code
            $this->portal_d_php_code->setupEditAttributes();
            if (!$this->portal_d_php_code->Raw) {
                $this->portal_d_php_code->AdvancedSearch->SearchValue = HtmlDecode($this->portal_d_php_code->AdvancedSearch->SearchValue);
            }
            $this->portal_d_php_code->EditValue = HtmlEncode($this->portal_d_php_code->AdvancedSearch->SearchValue);
            $this->portal_d_php_code->PlaceHolder = RemoveHtml($this->portal_d_php_code->caption());

            // portal_d_apc_query
            $this->portal_d_apc_query->setupEditAttributes();
            $this->portal_d_apc_query->EditValue = HtmlEncode($this->portal_d_apc_query->AdvancedSearch->SearchValue);
            $this->portal_d_apc_query->PlaceHolder = RemoveHtml($this->portal_d_apc_query->caption());

            // portal_d_mysql_query
            $this->portal_d_mysql_query->setupEditAttributes();
            if (!$this->portal_d_mysql_query->Raw) {
                $this->portal_d_mysql_query->AdvancedSearch->SearchValue = HtmlDecode($this->portal_d_mysql_query->AdvancedSearch->SearchValue);
            }
            $this->portal_d_mysql_query->EditValue = HtmlEncode($this->portal_d_mysql_query->AdvancedSearch->SearchValue);
            $this->portal_d_mysql_query->PlaceHolder = RemoveHtml($this->portal_d_mysql_query->caption());

            // portal_d_sp_query
            $this->portal_d_sp_query->setupEditAttributes();
            if (!$this->portal_d_sp_query->Raw) {
                $this->portal_d_sp_query->AdvancedSearch->SearchValue = HtmlDecode($this->portal_d_sp_query->AdvancedSearch->SearchValue);
            }
            $this->portal_d_sp_query->EditValue = HtmlEncode($this->portal_d_sp_query->AdvancedSearch->SearchValue);
            $this->portal_d_sp_query->PlaceHolder = RemoveHtml($this->portal_d_sp_query->caption());

            // portal_d_url_link
            $this->portal_d_url_link->setupEditAttributes();
            $this->portal_d_url_link->EditValue = HtmlEncode($this->portal_d_url_link->AdvancedSearch->SearchValue);
            $this->portal_d_url_link->PlaceHolder = RemoveHtml($this->portal_d_url_link->caption());

            // portal_c_php_code
            $this->portal_c_php_code->setupEditAttributes();
            if (!$this->portal_c_php_code->Raw) {
                $this->portal_c_php_code->AdvancedSearch->SearchValue = HtmlDecode($this->portal_c_php_code->AdvancedSearch->SearchValue);
            }
            $this->portal_c_php_code->EditValue = HtmlEncode($this->portal_c_php_code->AdvancedSearch->SearchValue);
            $curVal = strval($this->portal_c_php_code->AdvancedSearch->SearchValue);
            if ($curVal != "") {
                $this->portal_c_php_code->EditValue = $this->portal_c_php_code->lookupCacheOption($curVal);
                if ($this->portal_c_php_code->EditValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->portal_c_php_code->Lookup->getTable()->Fields["url"]->searchExpression(), "=", $curVal, $this->portal_c_php_code->Lookup->getTable()->Fields["url"]->searchDataType(), "");
                    $sqlWrk = $this->portal_c_php_code->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->portal_c_php_code->Lookup->renderViewRow($rswrk[0]);
                        $this->portal_c_php_code->EditValue = $this->portal_c_php_code->displayValue($arwrk);
                    } else {
                        $this->portal_c_php_code->EditValue = HtmlEncode($this->portal_c_php_code->AdvancedSearch->SearchValue);
                    }
                }
            } else {
                $this->portal_c_php_code->EditValue = null;
            }
            $this->portal_c_php_code->PlaceHolder = RemoveHtml($this->portal_c_php_code->caption());

            // mysql_query
            $this->mysql_query->setupEditAttributes();
            $this->mysql_query->EditValue = HtmlEncode($this->mysql_query->AdvancedSearch->SearchValue);
            $this->mysql_query->PlaceHolder = RemoveHtml($this->mysql_query->caption());

            // portal_c_mysql_query
            $this->portal_c_mysql_query->setupEditAttributes();
            if (!$this->portal_c_mysql_query->Raw) {
                $this->portal_c_mysql_query->AdvancedSearch->SearchValue = HtmlDecode($this->portal_c_mysql_query->AdvancedSearch->SearchValue);
            }
            $this->portal_c_mysql_query->EditValue = HtmlEncode($this->portal_c_mysql_query->AdvancedSearch->SearchValue);
            $curVal = strval($this->portal_c_mysql_query->AdvancedSearch->SearchValue);
            if ($curVal != "") {
                $this->portal_c_mysql_query->EditValue = $this->portal_c_mysql_query->lookupCacheOption($curVal);
                if ($this->portal_c_mysql_query->EditValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->portal_c_mysql_query->Lookup->getTable()->Fields["url"]->searchExpression(), "=", $curVal, $this->portal_c_mysql_query->Lookup->getTable()->Fields["url"]->searchDataType(), "");
                    $sqlWrk = $this->portal_c_mysql_query->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->portal_c_mysql_query->Lookup->renderViewRow($rswrk[0]);
                        $this->portal_c_mysql_query->EditValue = $this->portal_c_mysql_query->displayValue($arwrk);
                    } else {
                        $this->portal_c_mysql_query->EditValue = HtmlEncode($this->portal_c_mysql_query->AdvancedSearch->SearchValue);
                    }
                }
            } else {
                $this->portal_c_mysql_query->EditValue = null;
            }
            $this->portal_c_mysql_query->PlaceHolder = RemoveHtml($this->portal_c_mysql_query->caption());

            // sp_query
            $this->sp_query->setupEditAttributes();
            $this->sp_query->EditValue = HtmlEncode($this->sp_query->AdvancedSearch->SearchValue);
            $this->sp_query->PlaceHolder = RemoveHtml($this->sp_query->caption());

            // portal_c_sp_query
            $this->portal_c_sp_query->setupEditAttributes();
            if (!$this->portal_c_sp_query->Raw) {
                $this->portal_c_sp_query->AdvancedSearch->SearchValue = HtmlDecode($this->portal_c_sp_query->AdvancedSearch->SearchValue);
            }
            $this->portal_c_sp_query->EditValue = HtmlEncode($this->portal_c_sp_query->AdvancedSearch->SearchValue);
            $curVal = strval($this->portal_c_sp_query->AdvancedSearch->SearchValue);
            if ($curVal != "") {
                $this->portal_c_sp_query->EditValue = $this->portal_c_sp_query->lookupCacheOption($curVal);
                if ($this->portal_c_sp_query->EditValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->portal_c_sp_query->Lookup->getTable()->Fields["url"]->searchExpression(), "=", $curVal, $this->portal_c_sp_query->Lookup->getTable()->Fields["url"]->searchDataType(), "");
                    $sqlWrk = $this->portal_c_sp_query->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->portal_c_sp_query->Lookup->renderViewRow($rswrk[0]);
                        $this->portal_c_sp_query->EditValue = $this->portal_c_sp_query->displayValue($arwrk);
                    } else {
                        $this->portal_c_sp_query->EditValue = HtmlEncode($this->portal_c_sp_query->AdvancedSearch->SearchValue);
                    }
                }
            } else {
                $this->portal_c_sp_query->EditValue = null;
            }
            $this->portal_c_sp_query->PlaceHolder = RemoveHtml($this->portal_c_sp_query->caption());

            // apc_query
            $this->apc_query->setupEditAttributes();
            $this->apc_query->EditValue = HtmlEncode($this->apc_query->AdvancedSearch->SearchValue);
            $this->apc_query->PlaceHolder = RemoveHtml($this->apc_query->caption());

            // portal_c_apc_query
            $this->portal_c_apc_query->setupEditAttributes();
            $this->portal_c_apc_query->EditValue = HtmlEncode($this->portal_c_apc_query->AdvancedSearch->SearchValue);
            $this->portal_c_apc_query->PlaceHolder = RemoveHtml($this->portal_c_apc_query->caption());

            // url_link
            $this->url_link->setupEditAttributes();
            if (!$this->url_link->Raw) {
                $this->url_link->AdvancedSearch->SearchValue = HtmlDecode($this->url_link->AdvancedSearch->SearchValue);
            }
            $this->url_link->EditValue = HtmlEncode($this->url_link->AdvancedSearch->SearchValue);
            $this->url_link->PlaceHolder = RemoveHtml($this->url_link->caption());

            // u_type
            $this->u_type->setupEditAttributes();
            $this->u_type->EditValue = $this->u_type->AdvancedSearch->SearchValue;
            $this->u_type->PlaceHolder = RemoveHtml($this->u_type->caption());
        }
        if ($this->RowType == RowType::ADD || $this->RowType == RowType::EDIT || $this->RowType == RowType::SEARCH) { // Add/Edit/Search row
            $this->setupFieldTitles();
        }

        // Call Row Rendered event
        if ($this->RowType != RowType::AGGREGATEINIT) {
            $this->rowRendered();
        }
    }

    // Validate search
    protected function validateSearch()
    {
        // Check if validation required
        if (!Config("SERVER_VALIDATE")) {
            return true;
        }
        if (!CheckInteger($this->id->AdvancedSearch->SearchValue)) {
            $this->id->addErrorMessage($this->id->getErrorMessage(false));
        }
        if (!CheckDate($this->datetime->AdvancedSearch->SearchValue, $this->datetime->formatPattern())) {
            $this->datetime->addErrorMessage($this->datetime->getErrorMessage(false));
        }
        if (!CheckInteger($this->u_type->AdvancedSearch->SearchValue)) {
            $this->u_type->addErrorMessage($this->u_type->getErrorMessage(false));
        }

        // Return validate result
        $validateSearch = !$this->hasInvalidFields();

        // Call Form_CustomValidate event
        $formCustomError = "";
        $validateSearch = $validateSearch && $this->formCustomValidate($formCustomError);
        if ($formCustomError != "") {
            $this->setFailureMessage($formCustomError);
        }
        return $validateSearch;
    }

    // Load advanced search
    public function loadAdvancedSearch()
    {
        $this->id->AdvancedSearch->load();
        $this->datetime->AdvancedSearch->load();
        $this->version->AdvancedSearch->load();
        $this->php_code->AdvancedSearch->load();
        $this->portal_d_php_code->AdvancedSearch->load();
        $this->portal_d_apc_query->AdvancedSearch->load();
        $this->portal_d_mysql_query->AdvancedSearch->load();
        $this->portal_d_sp_query->AdvancedSearch->load();
        $this->portal_d_url_link->AdvancedSearch->load();
        $this->portal_c_php_code->AdvancedSearch->load();
        $this->mysql_query->AdvancedSearch->load();
        $this->portal_c_mysql_query->AdvancedSearch->load();
        $this->sp_query->AdvancedSearch->load();
        $this->portal_c_sp_query->AdvancedSearch->load();
        $this->apc_query->AdvancedSearch->load();
        $this->portal_c_apc_query->AdvancedSearch->load();
        $this->url_link->AdvancedSearch->load();
        $this->u_type->AdvancedSearch->load();
    }

    // Set up Breadcrumb
    protected function setupBreadcrumb()
    {
        global $Breadcrumb, $Language;
        $Breadcrumb = new Breadcrumb("db2");
        $url = CurrentUrl();
        $Breadcrumb->add("list", $this->TableVar, $this->addMasterUrl("portalversionlist"), "", $this->TableVar, true);
        $pageId = "search";
        $Breadcrumb->add("search", $pageId, $url);
    }

    // Set up multi pages
    protected function setupMultiPages()
    {
        $pages = new SubPages();
        $pages->Style = "tabs";
        if ($pages->isAccordion()) {
            $pages->Parent = "#accordion_" . $this->PageObjName;
        }
        $pages->add(0);
        $pages->add(1);
        $pages->add(2);
        $pages->add(3);
        $pages->add(4);
        $pages->add(5);
        $this->MultiPages = $pages;
    }

    // Setup lookup options
    public function setupLookupOptions($fld)
    {
        if ($fld->Lookup && $fld->Lookup->Options === null) {
            // Get default connection and filter
            $conn = $this->getConnection();
            $lookupFilter = "";

            // No need to check any more
            $fld->Lookup->Options = [];

            // Set up lookup SQL and connection
            switch ($fld->FieldVar) {
                case "x_portal_c_php_code":
                    break;
                case "x_portal_c_mysql_query":
                    break;
                case "x_portal_c_sp_query":
                    break;
                default:
                    $lookupFilter = "";
                    break;
            }

            // Always call to Lookup->getSql so that user can setup Lookup->Options in Lookup_Selecting server event
            $sql = $fld->Lookup->getSql(false, "", $lookupFilter, $this);

            // Set up lookup cache
            if (!$fld->hasLookupOptions() && $fld->UseLookupCache && $sql != "" && count($fld->Lookup->Options) == 0 && count($fld->Lookup->FilterFields) == 0) {
                $totalCnt = $this->getRecordCount($sql, $conn);
                if ($totalCnt > $fld->LookupCacheCount) { // Total count > cache count, do not cache
                    return;
                }
                $rows = $conn->executeQuery($sql)->fetchAll();
                $ar = [];
                foreach ($rows as $row) {
                    $row = $fld->Lookup->renderViewRow($row, Container($fld->Lookup->LinkTable));
                    $key = $row["lf"];
                    if (IsFloatType($fld->Type)) { // Handle float field
                        $key = (float)$key;
                    }
                    $ar[strval($key)] = $row;
                }
                $fld->Lookup->Options = $ar;
            }
        }
    }

    // Page Load event
    public function pageLoad()
    {
        //Log("Page Load");
    }

    // Page Unload event
    public function pageUnload()
    {
        //Log("Page Unload");
    }

    // Page Redirecting event
    public function pageRedirecting(&$url)
    {
        // Example:
        //$url = "your URL";
    }

    // Message Showing event
    // $type = ''|'success'|'failure'|'warning'
    public function messageShowing(&$msg, $type)
    {
        if ($type == "success") {
            //$msg = "your success message";
        } elseif ($type == "failure") {
            //$msg = "your failure message";
        } elseif ($type == "warning") {
            //$msg = "your warning message";
        } else {
            //$msg = "your message";
        }
    }

    // Page Render event
    public function pageRender()
    {
        //Log("Page Render");
    }

    // Page Data Rendering event
    public function pageDataRendering(&$header)
    {
        // Example:
        //$header = "your header";
    }

    // Page Data Rendered event
    public function pageDataRendered(&$footer)
    {
        // Example:
        //$footer = "your footer";
    }

    // Page Breaking event
    public function pageBreaking(&$break, &$content)
    {
        // Example:
        //$break = false; // Skip page break, or
        //$content = "<div style=\"break-after:page;\"></div>"; // Modify page break content
    }

    // Form Custom Validate event
    public function formCustomValidate(&$customError)
    {
        // Return error message in $customError
        return true;
    }
}
