<?php

namespace PHPMaker2024\fibre;

use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use Psr\Http\Message\ServerRequestInterface as Request;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Container\ContainerInterface;
use Slim\Routing\RouteCollectorProxy;
use Slim\App;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Closure;

/**
 * Page class
 */
class JournalSearch extends Journal
{
    use MessagesTrait;

    // Page ID
    public $PageID = "search";

    // Project ID
    public $ProjectID = PROJECT_ID;

    // Page object name
    public $PageObjName = "JournalSearch";

    // View file path
    public $View = null;

    // Title
    public $Title = null; // Title for <title> tag

    // Rendering View
    public $RenderingView = false;

    // CSS class/style
    public $CurrentPageName = "journalsearch";

    // Page headings
    public $Heading = "";
    public $Subheading = "";
    public $PageHeader;
    public $PageFooter;

    // Page layout
    public $UseLayout = true;

    // Page terminated
    private $terminated = false;

    // Page heading
    public function pageHeading()
    {
        global $Language;
        if ($this->Heading != "") {
            return $this->Heading;
        }
        if (method_exists($this, "tableCaption")) {
            return $this->tableCaption();
        }
        return "";
    }

    // Page subheading
    public function pageSubheading()
    {
        global $Language;
        if ($this->Subheading != "") {
            return $this->Subheading;
        }
        if ($this->TableName) {
            return $Language->phrase($this->PageID);
        }
        return "";
    }

    // Page name
    public function pageName()
    {
        return CurrentPageName();
    }

    // Page URL
    public function pageUrl($withArgs = true)
    {
        $route = GetRoute();
        $args = RemoveXss($route->getArguments());
        if (!$withArgs) {
            foreach ($args as $key => &$val) {
                $val = "";
            }
            unset($val);
        }
        return rtrim(UrlFor($route->getName(), $args), "/") . "?";
    }

    // Show Page Header
    public function showPageHeader()
    {
        $header = $this->PageHeader;
        $this->pageDataRendering($header);
        if ($header != "") { // Header exists, display
            echo '<div id="ew-page-header">' . $header . '</div>';
        }
    }

    // Show Page Footer
    public function showPageFooter()
    {
        $footer = $this->PageFooter;
        $this->pageDataRendered($footer);
        if ($footer != "") { // Footer exists, display
            echo '<div id="ew-page-footer">' . $footer . '</div>';
        }
    }

    // Set field visibility
    public function setVisibility()
    {
        $this->to_account->setVisibility();
        $this->from_account->setVisibility();
        $this->reference->setVisibility();
        $this->dated->setVisibility();
        $this->amount->setVisibility();
        $this->remarks->setVisibility();
        $this->uploads->setVisibility();
        $this->operator->setVisibility();
        $this->datetime->setVisibility();
        $this->id->setVisibility();
    }

    // Constructor
    public function __construct()
    {
        parent::__construct();
        global $Language, $DashboardReport, $DebugTimer, $UserTable;
        $this->TableVar = 'journal';
        $this->TableName = 'journal';

        // Table CSS class
        $this->TableClass = "table table-striped table-bordered table-hover table-sm ew-desktop-table ew-search-table";

        // Initialize
        $GLOBALS["Page"] = &$this;

        // Language object
        $Language = Container("app.language");

        // Table object (journal)
        if (!isset($GLOBALS["journal"]) || $GLOBALS["journal"]::class == PROJECT_NAMESPACE . "journal") {
            $GLOBALS["journal"] = &$this;
        }

        // Table name (for backward compatibility only)
        if (!defined(PROJECT_NAMESPACE . "TABLE_NAME")) {
            define(PROJECT_NAMESPACE . "TABLE_NAME", 'journal');
        }

        // Start timer
        $DebugTimer = Container("debug.timer");

        // Debug message
        LoadDebugMessage();

        // Open connection
        $GLOBALS["Conn"] ??= $this->getConnection();

        // User table object
        $UserTable = Container("usertable");
    }

    // Get content from stream
    public function getContents(): string
    {
        global $Response;
        return $Response?->getBody() ?? ob_get_clean();
    }

    // Is lookup
    public function isLookup()
    {
        return SameText(Route(0), Config("API_LOOKUP_ACTION"));
    }

    // Is AutoFill
    public function isAutoFill()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autofill");
    }

    // Is AutoSuggest
    public function isAutoSuggest()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autosuggest");
    }

    // Is modal lookup
    public function isModalLookup()
    {
        return $this->isLookup() && SameText(Post("ajax"), "modal");
    }

    // Is terminated
    public function isTerminated()
    {
        return $this->terminated;
    }

    /**
     * Terminate page
     *
     * @param string $url URL for direction
     * @return void
     */
    public function terminate($url = "")
    {
        if ($this->terminated) {
            return;
        }
        global $TempImages, $DashboardReport, $Response;

        // Page is terminated
        $this->terminated = true;

        // Page Unload event
        if (method_exists($this, "pageUnload")) {
            $this->pageUnload();
        }
        DispatchEvent(new PageUnloadedEvent($this), PageUnloadedEvent::NAME);
        if (!IsApi() && method_exists($this, "pageRedirecting")) {
            $this->pageRedirecting($url);
        }

        // Close connection
        CloseConnections();

        // Return for API
        if (IsApi()) {
            $res = $url === true;
            if (!$res) { // Show response for API
                $ar = array_merge($this->getMessages(), $url ? ["url" => GetUrl($url)] : []);
                WriteJson($ar);
            }
            $this->clearMessages(); // Clear messages for API request
            return;
        } else { // Check if response is JSON
            if (WithJsonResponse()) { // With JSON response
                $this->clearMessages();
                return;
            }
        }

        // Go to URL if specified
        if ($url != "") {
            if (!Config("DEBUG") && ob_get_length()) {
                ob_end_clean();
            }

            // Handle modal response
            if ($this->IsModal) { // Show as modal
                $pageName = GetPageName($url);
                $result = ["url" => GetUrl($url), "modal" => "1"];  // Assume return to modal for simplicity
                if (!SameString($pageName, GetPageName($this->getListUrl()))) { // Not List page
                    $result["caption"] = $this->getModalCaption($pageName);
                    $result["view"] = SameString($pageName, "journalview"); // If View page, no primary button
                } else { // List page
                    $result["error"] = $this->getFailureMessage(); // List page should not be shown as modal => error
                    $this->clearFailureMessage();
                }
                WriteJson($result);
            } else {
                SaveDebugMessage();
                Redirect(GetUrl($url));
            }
        }
        return; // Return to controller
    }

    // Get records from result set
    protected function getRecordsFromRecordset($rs, $current = false)
    {
        $rows = [];
        if (is_object($rs)) { // Result set
            while ($row = $rs->fetch()) {
                $this->loadRowValues($row); // Set up DbValue/CurrentValue
                $this->uploads->OldUploadPath = $this->uploads->getUploadPath(); // PHP
                $this->uploads->UploadPath = $this->uploads->OldUploadPath;
                $row = $this->getRecordFromArray($row);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        } elseif (is_array($rs)) {
            foreach ($rs as $ar) {
                $row = $this->getRecordFromArray($ar);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        }
        return $rows;
    }

    // Get record from array
    protected function getRecordFromArray($ar)
    {
        $row = [];
        if (is_array($ar)) {
            foreach ($ar as $fldname => $val) {
                if (array_key_exists($fldname, $this->Fields) && ($this->Fields[$fldname]->Visible || $this->Fields[$fldname]->IsPrimaryKey)) { // Primary key or Visible
                    $fld = &$this->Fields[$fldname];
                    if ($fld->HtmlTag == "FILE") { // Upload field
                        if (EmptyValue($val)) {
                            $row[$fldname] = null;
                        } else {
                            if ($fld->DataType == DataType::BLOB) {
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . $fld->Param . "/" . rawurlencode($this->getRecordKeyValue($ar))));
                                $row[$fldname] = ["type" => ContentType($val), "url" => $url, "name" => $fld->Param . ContentExtension($val)];
                            } elseif (!$fld->UploadMultiple || !ContainsString($val, Config("MULTIPLE_UPLOAD_SEPARATOR"))) { // Single file
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $val)));
                                $row[$fldname] = ["type" => MimeContentType($val), "url" => $url, "name" => $val];
                            } else { // Multiple files
                                $files = explode(Config("MULTIPLE_UPLOAD_SEPARATOR"), $val);
                                $ar = [];
                                foreach ($files as $file) {
                                    $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                        "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $file)));
                                    if (!EmptyValue($file)) {
                                        $ar[] = ["type" => MimeContentType($file), "url" => $url, "name" => $file];
                                    }
                                }
                                $row[$fldname] = $ar;
                            }
                        }
                    } else {
                        $row[$fldname] = $val;
                    }
                }
            }
        }
        return $row;
    }

    // Get record key value from array
    protected function getRecordKeyValue($ar)
    {
        $key = "";
        if (is_array($ar)) {
            $key .= @$ar['id'];
        }
        return $key;
    }

    /**
     * Hide fields for add/edit
     *
     * @return void
     */
    protected function hideFieldsForAddEdit()
    {
        if ($this->isAdd() || $this->isCopy() || $this->isGridAdd()) {
            $this->id->Visible = false;
        }
    }

    // Lookup data
    public function lookup(array $req = [], bool $response = true)
    {
        global $Language, $Security;

        // Get lookup object
        $fieldName = $req["field"] ?? null;
        if (!$fieldName) {
            return [];
        }
        $fld = $this->Fields[$fieldName];
        $lookup = $fld->Lookup;
        $name = $req["name"] ?? "";
        if (ContainsString($name, "query_builder_rule")) {
            $lookup->FilterFields = []; // Skip parent fields if any
        }

        // Get lookup parameters
        $lookupType = $req["ajax"] ?? "unknown";
        $pageSize = -1;
        $offset = -1;
        $searchValue = "";
        if (SameText($lookupType, "modal") || SameText($lookupType, "filter")) {
            $searchValue = $req["q"] ?? $req["sv"] ?? "";
            $pageSize = $req["n"] ?? $req["recperpage"] ?? 10;
        } elseif (SameText($lookupType, "autosuggest")) {
            $searchValue = $req["q"] ?? "";
            $pageSize = $req["n"] ?? -1;
            $pageSize = is_numeric($pageSize) ? (int)$pageSize : -1;
            if ($pageSize <= 0) {
                $pageSize = Config("AUTO_SUGGEST_MAX_ENTRIES");
            }
        }
        $start = $req["start"] ?? -1;
        $start = is_numeric($start) ? (int)$start : -1;
        $page = $req["page"] ?? -1;
        $page = is_numeric($page) ? (int)$page : -1;
        $offset = $start >= 0 ? $start : ($page > 0 && $pageSize > 0 ? ($page - 1) * $pageSize : 0);
        $userSelect = Decrypt($req["s"] ?? "");
        $userFilter = Decrypt($req["f"] ?? "");
        $userOrderBy = Decrypt($req["o"] ?? "");
        $keys = $req["keys"] ?? null;
        $lookup->LookupType = $lookupType; // Lookup type
        $lookup->FilterValues = []; // Clear filter values first
        if ($keys !== null) { // Selected records from modal
            if (is_array($keys)) {
                $keys = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $keys);
            }
            $lookup->FilterFields = []; // Skip parent fields if any
            $lookup->FilterValues[] = $keys; // Lookup values
            $pageSize = -1; // Show all records
        } else { // Lookup values
            $lookup->FilterValues[] = $req["v0"] ?? $req["lookupValue"] ?? "";
        }
        $cnt = is_array($lookup->FilterFields) ? count($lookup->FilterFields) : 0;
        for ($i = 1; $i <= $cnt; $i++) {
            $lookup->FilterValues[] = $req["v" . $i] ?? "";
        }
        $lookup->SearchValue = $searchValue;
        $lookup->PageSize = $pageSize;
        $lookup->Offset = $offset;
        if ($userSelect != "") {
            $lookup->UserSelect = $userSelect;
        }
        if ($userFilter != "") {
            $lookup->UserFilter = $userFilter;
        }
        if ($userOrderBy != "") {
            $lookup->UserOrderBy = $userOrderBy;
        }
        return $lookup->toJson($this, $response); // Use settings from current page
    }
    public $FormClassName = "ew-form ew-search-form";
    public $IsModal = false;
    public $IsMobileOrModal = false;

    /**
     * Page run
     *
     * @return void
     */
    public function run()
    {
        global $ExportType, $Language, $Security, $CurrentForm, $SkipHeaderFooter;

        // Is modal
        $this->IsModal = ConvertToBool(Param("modal"));
        $this->UseLayout = $this->UseLayout && !$this->IsModal;

        // Use layout
        $this->UseLayout = $this->UseLayout && ConvertToBool(Param(Config("PAGE_LAYOUT"), true));

        // View
        $this->View = Get(Config("VIEW"));

        // Load user profile
        if (IsLoggedIn()) {
            Profile()->setUserName(CurrentUserName())->loadFromStorage();
        }

        // Create form object
        $CurrentForm = new HttpForm();
        $this->CurrentAction = Param("action"); // Set up current action
        $this->setVisibility();

        // Set lookup cache
        if (!in_array($this->PageID, Config("LOOKUP_CACHE_PAGE_IDS"))) {
            $this->setUseLookupCache(false);
        }

        // Global Page Loading event (in userfn*.php)
        DispatchEvent(new PageLoadingEvent($this), PageLoadingEvent::NAME);

        // Page Load event
        if (method_exists($this, "pageLoad")) {
            $this->pageLoad();
        }

        // Hide fields for add/edit
        if (!$this->UseAjaxActions) {
            $this->hideFieldsForAddEdit();
        }
        // Use inline delete
        if ($this->UseAjaxActions) {
            $this->InlineDelete = true;
        }

        // Set up lookup cache
        $this->setupLookupOptions($this->to_account);
        $this->setupLookupOptions($this->from_account);
        $this->setupLookupOptions($this->operator);

        // Set up Breadcrumb
        $this->setupBreadcrumb();

        // Check modal
        if ($this->IsModal) {
            $SkipHeaderFooter = true;
        }
        $this->IsMobileOrModal = IsMobile() || $this->IsModal;

        // Get action
        $this->CurrentAction = Post("action");
        if ($this->isSearch()) {
            // Build search string for advanced search, remove blank field
            $this->loadSearchValues(); // Get search values
            $srchStr = $this->validateSearch() ? $this->buildAdvancedSearch() : "";
            if ($srchStr != "") {
                $srchStr = "journallist" . "?" . $srchStr;
                // Do not return Json for UseAjaxActions
                if ($this->IsModal && $this->UseAjaxActions) {
                    $this->IsModal = false;
                }
                $this->terminate($srchStr); // Go to list page
                return;
            }
        }

        // Restore search settings from Session
        if (!$this->hasInvalidFields()) {
            $this->loadAdvancedSearch();
        }

        // Render row for search
        $this->RowType = RowType::SEARCH;
        $this->resetAttributes();
        $this->renderRow();

        // Set LoginStatus / Page_Rendering / Page_Render
        if (!IsApi() && !$this->isTerminated()) {
            // Setup login status
            SetupLoginStatus();

            // Pass login status to client side
            SetClientVar("login", LoginStatus());

            // Global Page Rendering event (in userfn*.php)
            DispatchEvent(new PageRenderingEvent($this), PageRenderingEvent::NAME);

            // Page Render event
            if (method_exists($this, "pageRender")) {
                $this->pageRender();
            }

            // Render search option
            if (method_exists($this, "renderSearchOptions")) {
                $this->renderSearchOptions();
            }
        }
    }

    // Build advanced search
    protected function buildAdvancedSearch()
    {
        $srchUrl = "";
        $this->buildSearchUrl($srchUrl, $this->to_account); // to_account
        $this->buildSearchUrl($srchUrl, $this->from_account); // from_account
        $this->buildSearchUrl($srchUrl, $this->reference); // reference
        $this->buildSearchUrl($srchUrl, $this->dated); // dated
        $this->buildSearchUrl($srchUrl, $this->amount); // amount
        $this->buildSearchUrl($srchUrl, $this->remarks); // remarks
        $this->buildSearchUrl($srchUrl, $this->uploads); // uploads
        $this->buildSearchUrl($srchUrl, $this->operator); // operator
        $this->buildSearchUrl($srchUrl, $this->datetime); // datetime
        $this->buildSearchUrl($srchUrl, $this->id); // id
        if ($srchUrl != "") {
            $srchUrl .= "&";
        }
        $srchUrl .= "cmd=search";
        return $srchUrl;
    }

    // Build search URL
    protected function buildSearchUrl(&$url, $fld, $oprOnly = false)
    {
        global $CurrentForm;
        $wrk = "";
        $fldParm = $fld->Param;
        [
            "value" => $fldVal,
            "operator" => $fldOpr,
            "condition" => $fldCond,
            "value2" => $fldVal2,
            "operator2" => $fldOpr2
        ] = $CurrentForm->getSearchValues($fldParm);
        if (is_array($fldVal)) {
            $fldVal = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal);
        }
        if (is_array($fldVal2)) {
            $fldVal2 = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal2);
        }
        $fldDataType = $fld->DataType;
        $value = ConvertSearchValue($fldVal, $fldOpr, $fld); // For testing if numeric only
        $value2 = ConvertSearchValue($fldVal2, $fldOpr2, $fld); // For testing if numeric only
        $fldOpr = ConvertSearchOperator($fldOpr, $fld, $value);
        $fldOpr2 = ConvertSearchOperator($fldOpr2, $fld, $value2);
        if (in_array($fldOpr, ["BETWEEN", "NOT BETWEEN"])) {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld) && IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal != "" && $fldVal2 != "" && $isValidValue) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&y_" . $fldParm . "=" . urlencode($fldVal2) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            }
        } else {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld);
            if ($fldVal != "" && $isValidValue && IsValidOperator($fldOpr)) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            } elseif (in_array($fldOpr, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr != "" && $oprOnly && IsValidOperator($fldOpr))) {
                $wrk = "z_" . $fldParm . "=" . urlencode($fldOpr);
            }
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal2 != "" && $isValidValue && IsValidOperator($fldOpr2)) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "y_" . $fldParm . "=" . urlencode($fldVal2) . "&w_" . $fldParm . "=" . urlencode($fldOpr2);
            } elseif (in_array($fldOpr2, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr2 != "" && $oprOnly && IsValidOperator($fldOpr2))) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "w_" . $fldParm . "=" . urlencode($fldOpr2);
            }
        }
        if ($wrk != "") {
            if ($url != "") {
                $url .= "&";
            }
            $url .= $wrk;
        }
    }

    // Load search values for validation
    protected function loadSearchValues()
    {
        // Load search values
        $hasValue = false;

        // to_account
        if ($this->to_account->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // from_account
        if ($this->from_account->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // reference
        if ($this->reference->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // dated
        if ($this->dated->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // amount
        if ($this->amount->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // remarks
        if ($this->remarks->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // uploads
        if ($this->uploads->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // operator
        if ($this->operator->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // datetime
        if ($this->datetime->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // id
        if ($this->id->AdvancedSearch->get()) {
            $hasValue = true;
        }
        return $hasValue;
    }

    // Render row values based on field settings
    public function renderRow()
    {
        global $Security, $Language, $CurrentLanguage;

        // Initialize URLs

        // Call Row_Rendering event
        $this->rowRendering();

        // Common render codes for all row types

        // to_account
        $this->to_account->RowCssClass = "row";

        // from_account
        $this->from_account->RowCssClass = "row";

        // reference
        $this->reference->RowCssClass = "row";

        // dated
        $this->dated->RowCssClass = "row";

        // amount
        $this->amount->RowCssClass = "row";

        // remarks
        $this->remarks->RowCssClass = "row";

        // uploads
        $this->uploads->RowCssClass = "row";

        // operator
        $this->operator->RowCssClass = "row";

        // datetime
        $this->datetime->RowCssClass = "row";

        // id
        $this->id->RowCssClass = "row";

        // View row
        if ($this->RowType == RowType::VIEW) {
            // to_account
            if ($this->to_account->VirtualValue != "") {
                $this->to_account->ViewValue = $this->to_account->VirtualValue;
            } else {
                $curVal = strval($this->to_account->CurrentValue);
                if ($curVal != "") {
                    $this->to_account->ViewValue = $this->to_account->lookupCacheOption($curVal);
                    if ($this->to_account->ViewValue === null) { // Lookup from database
                        $filterWrk = SearchFilter($this->to_account->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->to_account->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                        $sqlWrk = $this->to_account->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                        $conn = Conn();
                        $config = $conn->getConfiguration();
                        $config->setResultCache($this->Cache);
                        $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                        $ari = count($rswrk);
                        if ($ari > 0) { // Lookup values found
                            $arwrk = $this->to_account->Lookup->renderViewRow($rswrk[0]);
                            $this->to_account->ViewValue = $this->to_account->displayValue($arwrk);
                        } else {
                            $this->to_account->ViewValue = FormatNumber($this->to_account->CurrentValue, $this->to_account->formatPattern());
                        }
                    }
                } else {
                    $this->to_account->ViewValue = null;
                }
            }

            // from_account
            if ($this->from_account->VirtualValue != "") {
                $this->from_account->ViewValue = $this->from_account->VirtualValue;
            } else {
                $curVal = strval($this->from_account->CurrentValue);
                if ($curVal != "") {
                    $this->from_account->ViewValue = $this->from_account->lookupCacheOption($curVal);
                    if ($this->from_account->ViewValue === null) { // Lookup from database
                        $filterWrk = SearchFilter($this->from_account->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->from_account->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                        $sqlWrk = $this->from_account->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                        $conn = Conn();
                        $config = $conn->getConfiguration();
                        $config->setResultCache($this->Cache);
                        $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                        $ari = count($rswrk);
                        if ($ari > 0) { // Lookup values found
                            $arwrk = $this->from_account->Lookup->renderViewRow($rswrk[0]);
                            $this->from_account->ViewValue = $this->from_account->displayValue($arwrk);
                        } else {
                            $this->from_account->ViewValue = FormatNumber($this->from_account->CurrentValue, $this->from_account->formatPattern());
                        }
                    }
                } else {
                    $this->from_account->ViewValue = null;
                }
            }

            // reference
            $this->reference->ViewValue = $this->reference->CurrentValue;

            // dated
            $this->dated->ViewValue = $this->dated->CurrentValue;
            $this->dated->ViewValue = FormatDateTime($this->dated->ViewValue, $this->dated->formatPattern());

            // amount
            $this->amount->ViewValue = $this->amount->CurrentValue;
            $this->amount->ViewValue = FormatNumber($this->amount->ViewValue, $this->amount->formatPattern());

            // remarks
            $this->remarks->ViewValue = $this->remarks->CurrentValue;

            // uploads
            $this->uploads->UploadPath = $this->uploads->getUploadPath(); // PHP
            if (!EmptyValue($this->uploads->Upload->DbValue)) {
                $this->uploads->ImageWidth = 0;
                $this->uploads->ImageHeight = 50;
                $this->uploads->ImageAlt = $this->uploads->alt();
                $this->uploads->ImageCssClass = "ew-image";
                $this->uploads->ViewValue = $this->uploads->Upload->DbValue;
            } else {
                $this->uploads->ViewValue = "";
            }

            // operator
            $curVal = strval($this->operator->CurrentValue);
            if ($curVal != "") {
                $this->operator->ViewValue = $this->operator->lookupCacheOption($curVal);
                if ($this->operator->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->operator->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->operator->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                    $sqlWrk = $this->operator->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->operator->Lookup->renderViewRow($rswrk[0]);
                        $this->operator->ViewValue = $this->operator->displayValue($arwrk);
                    } else {
                        $this->operator->ViewValue = FormatNumber($this->operator->CurrentValue, $this->operator->formatPattern());
                    }
                }
            } else {
                $this->operator->ViewValue = null;
            }

            // datetime
            $this->datetime->ViewValue = $this->datetime->CurrentValue;
            $this->datetime->ViewValue = FormatDateTime($this->datetime->ViewValue, $this->datetime->formatPattern());

            // id
            $this->id->ViewValue = $this->id->CurrentValue;

            // to_account
            $this->to_account->HrefValue = "";
            $this->to_account->TooltipValue = "";

            // from_account
            $this->from_account->HrefValue = "";
            $this->from_account->TooltipValue = "";

            // reference
            $this->reference->HrefValue = "";
            $this->reference->TooltipValue = "";

            // dated
            $this->dated->HrefValue = "";
            $this->dated->TooltipValue = "";

            // amount
            $this->amount->HrefValue = "";
            $this->amount->TooltipValue = "";

            // remarks
            $this->remarks->HrefValue = "";
            $this->remarks->TooltipValue = "";

            // uploads
            $this->uploads->UploadPath = $this->uploads->getUploadPath(); // PHP
            if (!EmptyValue($this->uploads->Upload->DbValue)) {
                $this->uploads->HrefValue = "%u"; // Add prefix/suffix
                $this->uploads->LinkAttrs["target"] = ""; // Add target
                if ($this->isExport()) {
                    $this->uploads->HrefValue = FullUrl($this->uploads->HrefValue, "href");
                }
            } else {
                $this->uploads->HrefValue = "";
            }
            $this->uploads->ExportHrefValue = $this->uploads->UploadPath . $this->uploads->Upload->DbValue;
            $this->uploads->TooltipValue = "";
            if ($this->uploads->UseColorbox) {
                if (EmptyValue($this->uploads->TooltipValue)) {
                    $this->uploads->LinkAttrs["title"] = $Language->phrase("ViewImageGallery");
                }
                $this->uploads->LinkAttrs["data-rel"] = "journal_x_uploads";
                $this->uploads->LinkAttrs->appendClass("ew-lightbox");
            }

            // operator
            $this->operator->HrefValue = "";
            $this->operator->TooltipValue = "";

            // datetime
            $this->datetime->HrefValue = "";
            $this->datetime->TooltipValue = "";

            // id
            $this->id->HrefValue = "";
            $this->id->TooltipValue = "";
        } elseif ($this->RowType == RowType::SEARCH) {
            // to_account
            $curVal = trim(strval($this->to_account->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->to_account->AdvancedSearch->ViewValue = $this->to_account->lookupCacheOption($curVal);
            } else {
                $this->to_account->AdvancedSearch->ViewValue = $this->to_account->Lookup !== null && is_array($this->to_account->lookupOptions()) && count($this->to_account->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->to_account->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->to_account->EditValue = array_values($this->to_account->lookupOptions());
                if ($this->to_account->AdvancedSearch->ViewValue == "") {
                    $this->to_account->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->to_account->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $this->to_account->AdvancedSearch->SearchValue, $this->to_account->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                }
                $sqlWrk = $this->to_account->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                if ($ari > 0) { // Lookup values found
                    $arwrk = $this->to_account->Lookup->renderViewRow($rswrk[0]);
                    $this->to_account->AdvancedSearch->ViewValue = $this->to_account->displayValue($arwrk);
                } else {
                    $this->to_account->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
                $arwrk = $rswrk;
                foreach ($arwrk as &$row) {
                    $row = $this->to_account->Lookup->renderViewRow($row);
                }
                $this->to_account->EditValue = $arwrk;
            }
            $this->to_account->PlaceHolder = RemoveHtml($this->to_account->caption());

            // from_account
            $curVal = trim(strval($this->from_account->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->from_account->AdvancedSearch->ViewValue = $this->from_account->lookupCacheOption($curVal);
            } else {
                $this->from_account->AdvancedSearch->ViewValue = $this->from_account->Lookup !== null && is_array($this->from_account->lookupOptions()) && count($this->from_account->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->from_account->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->from_account->EditValue = array_values($this->from_account->lookupOptions());
                if ($this->from_account->AdvancedSearch->ViewValue == "") {
                    $this->from_account->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->from_account->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $this->from_account->AdvancedSearch->SearchValue, $this->from_account->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                }
                $sqlWrk = $this->from_account->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                if ($ari > 0) { // Lookup values found
                    $arwrk = $this->from_account->Lookup->renderViewRow($rswrk[0]);
                    $this->from_account->AdvancedSearch->ViewValue = $this->from_account->displayValue($arwrk);
                } else {
                    $this->from_account->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
                $arwrk = $rswrk;
                foreach ($arwrk as &$row) {
                    $row = $this->from_account->Lookup->renderViewRow($row);
                }
                $this->from_account->EditValue = $arwrk;
            }
            $this->from_account->PlaceHolder = RemoveHtml($this->from_account->caption());

            // reference
            $this->reference->setupEditAttributes();
            if (!$this->reference->Raw) {
                $this->reference->AdvancedSearch->SearchValue = HtmlDecode($this->reference->AdvancedSearch->SearchValue);
            }
            $this->reference->EditValue = HtmlEncode($this->reference->AdvancedSearch->SearchValue);
            $this->reference->PlaceHolder = RemoveHtml($this->reference->caption());

            // dated
            $this->dated->setupEditAttributes();
            $this->dated->EditValue = HtmlEncode(FormatDateTime(UnFormatDateTime($this->dated->AdvancedSearch->SearchValue, $this->dated->formatPattern()), $this->dated->formatPattern()));
            $this->dated->PlaceHolder = RemoveHtml($this->dated->caption());
            $this->dated->setupEditAttributes();
            $this->dated->EditValue2 = HtmlEncode(FormatDateTime(UnFormatDateTime($this->dated->AdvancedSearch->SearchValue2, $this->dated->formatPattern()), $this->dated->formatPattern()));
            $this->dated->PlaceHolder = RemoveHtml($this->dated->caption());

            // amount
            $this->amount->setupEditAttributes();
            $this->amount->EditValue = $this->amount->AdvancedSearch->SearchValue;
            $this->amount->PlaceHolder = RemoveHtml($this->amount->caption());

            // remarks
            $this->remarks->setupEditAttributes();
            if (!$this->remarks->Raw) {
                $this->remarks->AdvancedSearch->SearchValue = HtmlDecode($this->remarks->AdvancedSearch->SearchValue);
            }
            $this->remarks->EditValue = HtmlEncode($this->remarks->AdvancedSearch->SearchValue);
            $this->remarks->PlaceHolder = RemoveHtml($this->remarks->caption());

            // uploads
            $this->uploads->setupEditAttributes();
            if (!$this->uploads->Raw) {
                $this->uploads->AdvancedSearch->SearchValue = HtmlDecode($this->uploads->AdvancedSearch->SearchValue);
            }
            $this->uploads->EditValue = HtmlEncode($this->uploads->AdvancedSearch->SearchValue);
            $this->uploads->PlaceHolder = RemoveHtml($this->uploads->caption());

            // operator
            $this->operator->setupEditAttributes();
            $curVal = trim(strval($this->operator->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->operator->AdvancedSearch->ViewValue = $this->operator->lookupCacheOption($curVal);
            } else {
                $this->operator->AdvancedSearch->ViewValue = $this->operator->Lookup !== null && is_array($this->operator->lookupOptions()) && count($this->operator->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->operator->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->operator->EditValue = array_values($this->operator->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->operator->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $this->operator->AdvancedSearch->SearchValue, $this->operator->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                }
                $sqlWrk = $this->operator->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->operator->EditValue = $arwrk;
            }
            $this->operator->PlaceHolder = RemoveHtml($this->operator->caption());

            // datetime
            $this->datetime->setupEditAttributes();
            $this->datetime->EditValue = HtmlEncode(FormatDateTime(UnFormatDateTime($this->datetime->AdvancedSearch->SearchValue, $this->datetime->formatPattern()), $this->datetime->formatPattern()));
            $this->datetime->PlaceHolder = RemoveHtml($this->datetime->caption());
            $this->datetime->setupEditAttributes();
            $this->datetime->EditValue2 = HtmlEncode(FormatDateTime(UnFormatDateTime($this->datetime->AdvancedSearch->SearchValue2, $this->datetime->formatPattern()), $this->datetime->formatPattern()));
            $this->datetime->PlaceHolder = RemoveHtml($this->datetime->caption());

            // id
            $this->id->setupEditAttributes();
            $this->id->EditValue = $this->id->AdvancedSearch->SearchValue;
            $this->id->PlaceHolder = RemoveHtml($this->id->caption());
        }
        if ($this->RowType == RowType::ADD || $this->RowType == RowType::EDIT || $this->RowType == RowType::SEARCH) { // Add/Edit/Search row
            $this->setupFieldTitles();
        }

        // Call Row Rendered event
        if ($this->RowType != RowType::AGGREGATEINIT) {
            $this->rowRendered();
        }
    }

    // Validate search
    protected function validateSearch()
    {
        // Check if validation required
        if (!Config("SERVER_VALIDATE")) {
            return true;
        }
        if (!CheckDate($this->dated->AdvancedSearch->SearchValue, $this->dated->formatPattern())) {
            $this->dated->addErrorMessage($this->dated->getErrorMessage(false));
        }
        if (!CheckDate($this->dated->AdvancedSearch->SearchValue2, $this->dated->formatPattern())) {
            $this->dated->addErrorMessage($this->dated->getErrorMessage(false));
        }
        if (!CheckNumber($this->amount->AdvancedSearch->SearchValue)) {
            $this->amount->addErrorMessage($this->amount->getErrorMessage(false));
        }
        if (!CheckDate($this->datetime->AdvancedSearch->SearchValue, $this->datetime->formatPattern())) {
            $this->datetime->addErrorMessage($this->datetime->getErrorMessage(false));
        }
        if (!CheckDate($this->datetime->AdvancedSearch->SearchValue2, $this->datetime->formatPattern())) {
            $this->datetime->addErrorMessage($this->datetime->getErrorMessage(false));
        }
        if (!CheckInteger($this->id->AdvancedSearch->SearchValue)) {
            $this->id->addErrorMessage($this->id->getErrorMessage(false));
        }

        // Return validate result
        $validateSearch = !$this->hasInvalidFields();

        // Call Form_CustomValidate event
        $formCustomError = "";
        $validateSearch = $validateSearch && $this->formCustomValidate($formCustomError);
        if ($formCustomError != "") {
            $this->setFailureMessage($formCustomError);
        }
        return $validateSearch;
    }

    // Load advanced search
    public function loadAdvancedSearch()
    {
        $this->to_account->AdvancedSearch->load();
        $this->from_account->AdvancedSearch->load();
        $this->reference->AdvancedSearch->load();
        $this->dated->AdvancedSearch->load();
        $this->amount->AdvancedSearch->load();
        $this->remarks->AdvancedSearch->load();
        $this->uploads->AdvancedSearch->load();
        $this->operator->AdvancedSearch->load();
        $this->datetime->AdvancedSearch->load();
        $this->id->AdvancedSearch->load();
    }

    // Set up Breadcrumb
    protected function setupBreadcrumb()
    {
        global $Breadcrumb, $Language;
        $Breadcrumb = new Breadcrumb("db2");
        $url = CurrentUrl();
        $Breadcrumb->add("list", $this->TableVar, $this->addMasterUrl("journallist"), "", $this->TableVar, true);
        $pageId = "search";
        $Breadcrumb->add("search", $pageId, $url);
    }

    // Setup lookup options
    public function setupLookupOptions($fld)
    {
        if ($fld->Lookup && $fld->Lookup->Options === null) {
            // Get default connection and filter
            $conn = $this->getConnection();
            $lookupFilter = "";

            // No need to check any more
            $fld->Lookup->Options = [];

            // Set up lookup SQL and connection
            switch ($fld->FieldVar) {
                case "x_to_account":
                    break;
                case "x_from_account":
                    break;
                case "x_operator":
                    break;
                default:
                    $lookupFilter = "";
                    break;
            }

            // Always call to Lookup->getSql so that user can setup Lookup->Options in Lookup_Selecting server event
            $sql = $fld->Lookup->getSql(false, "", $lookupFilter, $this);

            // Set up lookup cache
            if (!$fld->hasLookupOptions() && $fld->UseLookupCache && $sql != "" && count($fld->Lookup->Options) == 0 && count($fld->Lookup->FilterFields) == 0) {
                $totalCnt = $this->getRecordCount($sql, $conn);
                if ($totalCnt > $fld->LookupCacheCount) { // Total count > cache count, do not cache
                    return;
                }
                $rows = $conn->executeQuery($sql)->fetchAll();
                $ar = [];
                foreach ($rows as $row) {
                    $row = $fld->Lookup->renderViewRow($row, Container($fld->Lookup->LinkTable));
                    $key = $row["lf"];
                    if (IsFloatType($fld->Type)) { // Handle float field
                        $key = (float)$key;
                    }
                    $ar[strval($key)] = $row;
                }
                $fld->Lookup->Options = $ar;
            }
        }
    }

    // Page Load event
    public function pageLoad()
    {
        //Log("Page Load");
    }

    // Page Unload event
    public function pageUnload()
    {
        //Log("Page Unload");
    }

    // Page Redirecting event
    public function pageRedirecting(&$url)
    {
        // Example:
        //$url = "your URL";
    }

    // Message Showing event
    // $type = ''|'success'|'failure'|'warning'
    public function messageShowing(&$msg, $type)
    {
        if ($type == "success") {
            //$msg = "your success message";
        } elseif ($type == "failure") {
            //$msg = "your failure message";
        } elseif ($type == "warning") {
            //$msg = "your warning message";
        } else {
            //$msg = "your message";
        }
    }

    // Page Render event
    public function pageRender()
    {
        //Log("Page Render");
    }

    // Page Data Rendering event
    public function pageDataRendering(&$header)
    {
        // Example:
        //$header = "your header";
    }

    // Page Data Rendered event
    public function pageDataRendered(&$footer)
    {
        // Example:
        //$footer = "your footer";
    }

    // Page Breaking event
    public function pageBreaking(&$break, &$content)
    {
        // Example:
        //$break = false; // Skip page break, or
        //$content = "<div style=\"break-after:page;\"></div>"; // Modify page break content
    }

    // Form Custom Validate event
    public function formCustomValidate(&$customError)
    {
        // Return error message in $customError
        return true;
    }
}
