<?php

namespace PHPMaker2024\fibre;

use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use Psr\Http\Message\ServerRequestInterface as Request;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Container\ContainerInterface;
use Slim\Routing\RouteCollectorProxy;
use Slim\App;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Closure;

/**
 * Page class
 */
class HelpSearch extends Help
{
    use MessagesTrait;

    // Page ID
    public $PageID = "search";

    // Project ID
    public $ProjectID = PROJECT_ID;

    // Page object name
    public $PageObjName = "HelpSearch";

    // View file path
    public $View = null;

    // Title
    public $Title = null; // Title for <title> tag

    // Rendering View
    public $RenderingView = false;

    // CSS class/style
    public $CurrentPageName = "helpsearch";

    // Page headings
    public $Heading = "";
    public $Subheading = "";
    public $PageHeader;
    public $PageFooter;

    // Page layout
    public $UseLayout = true;

    // Page terminated
    private $terminated = false;

    // Page heading
    public function pageHeading()
    {
        global $Language;
        if ($this->Heading != "") {
            return $this->Heading;
        }
        if (method_exists($this, "tableCaption")) {
            return $this->tableCaption();
        }
        return "";
    }

    // Page subheading
    public function pageSubheading()
    {
        global $Language;
        if ($this->Subheading != "") {
            return $this->Subheading;
        }
        if ($this->TableName) {
            return $Language->phrase($this->PageID);
        }
        return "";
    }

    // Page name
    public function pageName()
    {
        return CurrentPageName();
    }

    // Page URL
    public function pageUrl($withArgs = true)
    {
        $route = GetRoute();
        $args = RemoveXss($route->getArguments());
        if (!$withArgs) {
            foreach ($args as $key => &$val) {
                $val = "";
            }
            unset($val);
        }
        return rtrim(UrlFor($route->getName(), $args), "/") . "?";
    }

    // Show Page Header
    public function showPageHeader()
    {
        $header = $this->PageHeader;
        $this->pageDataRendering($header);
        if ($header != "") { // Header exists, display
            echo '<div id="ew-page-header">' . $header . '</div>';
        }
    }

    // Show Page Footer
    public function showPageFooter()
    {
        $footer = $this->PageFooter;
        $this->pageDataRendered($footer);
        if ($footer != "") { // Footer exists, display
            echo '<div id="ew-page-footer">' . $footer . '</div>';
        }
    }

    // Set field visibility
    public function setVisibility()
    {
        $this->id->setVisibility();
        $this->level->setVisibility();
        $this->menu_group->setVisibility();
        $this->menu_description->setVisibility();
        $this->table_name->setVisibility();
        $this->table_caption->setVisibility();
        $this->field_name->setVisibility();
        $this->field_caption->setVisibility();
        $this->field_description->setVisibility();
        $this->table_description->setVisibility();
        $this->system_notes->Visible = false;
        $this->portal_notes->setVisibility();
        $this->admin_notes->Visible = false;
        $this->_visible->setVisibility();
        $this->operator->setVisibility();
        $this->datetime->setVisibility();
    }

    // Constructor
    public function __construct()
    {
        parent::__construct();
        global $Language, $DashboardReport, $DebugTimer, $UserTable;
        $this->TableVar = 'help';
        $this->TableName = 'help';

        // Table CSS class
        $this->TableClass = "table table-striped table-bordered table-hover table-sm ew-desktop-table ew-search-table";

        // Initialize
        $GLOBALS["Page"] = &$this;

        // Language object
        $Language = Container("app.language");

        // Table object (help)
        if (!isset($GLOBALS["help"]) || $GLOBALS["help"]::class == PROJECT_NAMESPACE . "help") {
            $GLOBALS["help"] = &$this;
        }

        // Table name (for backward compatibility only)
        if (!defined(PROJECT_NAMESPACE . "TABLE_NAME")) {
            define(PROJECT_NAMESPACE . "TABLE_NAME", 'help');
        }

        // Start timer
        $DebugTimer = Container("debug.timer");

        // Debug message
        LoadDebugMessage();

        // Open connection
        $GLOBALS["Conn"] ??= $this->getConnection();

        // User table object
        $UserTable = Container("usertable");
    }

    // Get content from stream
    public function getContents(): string
    {
        global $Response;
        return $Response?->getBody() ?? ob_get_clean();
    }

    // Is lookup
    public function isLookup()
    {
        return SameText(Route(0), Config("API_LOOKUP_ACTION"));
    }

    // Is AutoFill
    public function isAutoFill()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autofill");
    }

    // Is AutoSuggest
    public function isAutoSuggest()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autosuggest");
    }

    // Is modal lookup
    public function isModalLookup()
    {
        return $this->isLookup() && SameText(Post("ajax"), "modal");
    }

    // Is terminated
    public function isTerminated()
    {
        return $this->terminated;
    }

    /**
     * Terminate page
     *
     * @param string $url URL for direction
     * @return void
     */
    public function terminate($url = "")
    {
        if ($this->terminated) {
            return;
        }
        global $TempImages, $DashboardReport, $Response;

        // Page is terminated
        $this->terminated = true;

        // Page Unload event
        if (method_exists($this, "pageUnload")) {
            $this->pageUnload();
        }
        DispatchEvent(new PageUnloadedEvent($this), PageUnloadedEvent::NAME);
        if (!IsApi() && method_exists($this, "pageRedirecting")) {
            $this->pageRedirecting($url);
        }

        // Close connection
        CloseConnections();

        // Return for API
        if (IsApi()) {
            $res = $url === true;
            if (!$res) { // Show response for API
                $ar = array_merge($this->getMessages(), $url ? ["url" => GetUrl($url)] : []);
                WriteJson($ar);
            }
            $this->clearMessages(); // Clear messages for API request
            return;
        } else { // Check if response is JSON
            if (WithJsonResponse()) { // With JSON response
                $this->clearMessages();
                return;
            }
        }

        // Go to URL if specified
        if ($url != "") {
            if (!Config("DEBUG") && ob_get_length()) {
                ob_end_clean();
            }

            // Handle modal response
            if ($this->IsModal) { // Show as modal
                $pageName = GetPageName($url);
                $result = ["url" => GetUrl($url), "modal" => "1"];  // Assume return to modal for simplicity
                if (!SameString($pageName, GetPageName($this->getListUrl()))) { // Not List page
                    $result["caption"] = $this->getModalCaption($pageName);
                    $result["view"] = SameString($pageName, "helpview"); // If View page, no primary button
                } else { // List page
                    $result["error"] = $this->getFailureMessage(); // List page should not be shown as modal => error
                    $this->clearFailureMessage();
                }
                WriteJson($result);
            } else {
                SaveDebugMessage();
                Redirect(GetUrl($url));
            }
        }
        return; // Return to controller
    }

    // Get records from result set
    protected function getRecordsFromRecordset($rs, $current = false)
    {
        $rows = [];
        if (is_object($rs)) { // Result set
            while ($row = $rs->fetch()) {
                $this->loadRowValues($row); // Set up DbValue/CurrentValue
                $row = $this->getRecordFromArray($row);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        } elseif (is_array($rs)) {
            foreach ($rs as $ar) {
                $row = $this->getRecordFromArray($ar);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        }
        return $rows;
    }

    // Get record from array
    protected function getRecordFromArray($ar)
    {
        $row = [];
        if (is_array($ar)) {
            foreach ($ar as $fldname => $val) {
                if (array_key_exists($fldname, $this->Fields) && ($this->Fields[$fldname]->Visible || $this->Fields[$fldname]->IsPrimaryKey)) { // Primary key or Visible
                    $fld = &$this->Fields[$fldname];
                    if ($fld->HtmlTag == "FILE") { // Upload field
                        if (EmptyValue($val)) {
                            $row[$fldname] = null;
                        } else {
                            if ($fld->DataType == DataType::BLOB) {
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . $fld->Param . "/" . rawurlencode($this->getRecordKeyValue($ar))));
                                $row[$fldname] = ["type" => ContentType($val), "url" => $url, "name" => $fld->Param . ContentExtension($val)];
                            } elseif (!$fld->UploadMultiple || !ContainsString($val, Config("MULTIPLE_UPLOAD_SEPARATOR"))) { // Single file
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $val)));
                                $row[$fldname] = ["type" => MimeContentType($val), "url" => $url, "name" => $val];
                            } else { // Multiple files
                                $files = explode(Config("MULTIPLE_UPLOAD_SEPARATOR"), $val);
                                $ar = [];
                                foreach ($files as $file) {
                                    $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                        "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $file)));
                                    if (!EmptyValue($file)) {
                                        $ar[] = ["type" => MimeContentType($file), "url" => $url, "name" => $file];
                                    }
                                }
                                $row[$fldname] = $ar;
                            }
                        }
                    } else {
                        $row[$fldname] = $val;
                    }
                }
            }
        }
        return $row;
    }

    // Get record key value from array
    protected function getRecordKeyValue($ar)
    {
        $key = "";
        if (is_array($ar)) {
            $key .= @$ar['id'];
        }
        return $key;
    }

    /**
     * Hide fields for add/edit
     *
     * @return void
     */
    protected function hideFieldsForAddEdit()
    {
        if ($this->isAdd() || $this->isCopy() || $this->isGridAdd()) {
            $this->id->Visible = false;
        }
    }

    // Lookup data
    public function lookup(array $req = [], bool $response = true)
    {
        global $Language, $Security;

        // Get lookup object
        $fieldName = $req["field"] ?? null;
        if (!$fieldName) {
            return [];
        }
        $fld = $this->Fields[$fieldName];
        $lookup = $fld->Lookup;
        $name = $req["name"] ?? "";
        if (ContainsString($name, "query_builder_rule")) {
            $lookup->FilterFields = []; // Skip parent fields if any
        }

        // Get lookup parameters
        $lookupType = $req["ajax"] ?? "unknown";
        $pageSize = -1;
        $offset = -1;
        $searchValue = "";
        if (SameText($lookupType, "modal") || SameText($lookupType, "filter")) {
            $searchValue = $req["q"] ?? $req["sv"] ?? "";
            $pageSize = $req["n"] ?? $req["recperpage"] ?? 10;
        } elseif (SameText($lookupType, "autosuggest")) {
            $searchValue = $req["q"] ?? "";
            $pageSize = $req["n"] ?? -1;
            $pageSize = is_numeric($pageSize) ? (int)$pageSize : -1;
            if ($pageSize <= 0) {
                $pageSize = Config("AUTO_SUGGEST_MAX_ENTRIES");
            }
        }
        $start = $req["start"] ?? -1;
        $start = is_numeric($start) ? (int)$start : -1;
        $page = $req["page"] ?? -1;
        $page = is_numeric($page) ? (int)$page : -1;
        $offset = $start >= 0 ? $start : ($page > 0 && $pageSize > 0 ? ($page - 1) * $pageSize : 0);
        $userSelect = Decrypt($req["s"] ?? "");
        $userFilter = Decrypt($req["f"] ?? "");
        $userOrderBy = Decrypt($req["o"] ?? "");
        $keys = $req["keys"] ?? null;
        $lookup->LookupType = $lookupType; // Lookup type
        $lookup->FilterValues = []; // Clear filter values first
        if ($keys !== null) { // Selected records from modal
            if (is_array($keys)) {
                $keys = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $keys);
            }
            $lookup->FilterFields = []; // Skip parent fields if any
            $lookup->FilterValues[] = $keys; // Lookup values
            $pageSize = -1; // Show all records
        } else { // Lookup values
            $lookup->FilterValues[] = $req["v0"] ?? $req["lookupValue"] ?? "";
        }
        $cnt = is_array($lookup->FilterFields) ? count($lookup->FilterFields) : 0;
        for ($i = 1; $i <= $cnt; $i++) {
            $lookup->FilterValues[] = $req["v" . $i] ?? "";
        }
        $lookup->SearchValue = $searchValue;
        $lookup->PageSize = $pageSize;
        $lookup->Offset = $offset;
        if ($userSelect != "") {
            $lookup->UserSelect = $userSelect;
        }
        if ($userFilter != "") {
            $lookup->UserFilter = $userFilter;
        }
        if ($userOrderBy != "") {
            $lookup->UserOrderBy = $userOrderBy;
        }
        return $lookup->toJson($this, $response); // Use settings from current page
    }
    public $FormClassName = "ew-form ew-search-form";
    public $IsModal = false;
    public $IsMobileOrModal = false;

    /**
     * Page run
     *
     * @return void
     */
    public function run()
    {
        global $ExportType, $Language, $Security, $CurrentForm, $SkipHeaderFooter;

        // Is modal
        $this->IsModal = ConvertToBool(Param("modal"));
        $this->UseLayout = $this->UseLayout && !$this->IsModal;

        // Use layout
        $this->UseLayout = $this->UseLayout && ConvertToBool(Param(Config("PAGE_LAYOUT"), true));

        // View
        $this->View = Get(Config("VIEW"));

        // Load user profile
        if (IsLoggedIn()) {
            Profile()->setUserName(CurrentUserName())->loadFromStorage();
        }

        // Create form object
        $CurrentForm = new HttpForm();
        $this->CurrentAction = Param("action"); // Set up current action
        $this->setVisibility();

        // Set lookup cache
        if (!in_array($this->PageID, Config("LOOKUP_CACHE_PAGE_IDS"))) {
            $this->setUseLookupCache(false);
        }

        // Global Page Loading event (in userfn*.php)
        DispatchEvent(new PageLoadingEvent($this), PageLoadingEvent::NAME);

        // Page Load event
        if (method_exists($this, "pageLoad")) {
            $this->pageLoad();
        }

        // Hide fields for add/edit
        if (!$this->UseAjaxActions) {
            $this->hideFieldsForAddEdit();
        }
        // Use inline delete
        if ($this->UseAjaxActions) {
            $this->InlineDelete = true;
        }

        // Set up lookup cache
        $this->setupLookupOptions($this->level);
        $this->setupLookupOptions($this->_visible);
        $this->setupLookupOptions($this->operator);

        // Set up Breadcrumb
        $this->setupBreadcrumb();

        // Check modal
        if ($this->IsModal) {
            $SkipHeaderFooter = true;
        }
        $this->IsMobileOrModal = IsMobile() || $this->IsModal;

        // Get action
        $this->CurrentAction = Post("action");
        if ($this->isSearch()) {
            // Build search string for advanced search, remove blank field
            $this->loadSearchValues(); // Get search values
            $srchStr = $this->validateSearch() ? $this->buildAdvancedSearch() : "";
            if ($srchStr != "") {
                $srchStr = "helplist" . "?" . $srchStr;
                // Do not return Json for UseAjaxActions
                if ($this->IsModal && $this->UseAjaxActions) {
                    $this->IsModal = false;
                }
                $this->terminate($srchStr); // Go to list page
                return;
            }
        }

        // Restore search settings from Session
        if (!$this->hasInvalidFields()) {
            $this->loadAdvancedSearch();
        }

        // Render row for search
        $this->RowType = RowType::SEARCH;
        $this->resetAttributes();
        $this->renderRow();

        // Set LoginStatus / Page_Rendering / Page_Render
        if (!IsApi() && !$this->isTerminated()) {
            // Setup login status
            SetupLoginStatus();

            // Pass login status to client side
            SetClientVar("login", LoginStatus());

            // Global Page Rendering event (in userfn*.php)
            DispatchEvent(new PageRenderingEvent($this), PageRenderingEvent::NAME);

            // Page Render event
            if (method_exists($this, "pageRender")) {
                $this->pageRender();
            }

            // Render search option
            if (method_exists($this, "renderSearchOptions")) {
                $this->renderSearchOptions();
            }
        }
    }

    // Build advanced search
    protected function buildAdvancedSearch()
    {
        $srchUrl = "";
        $this->buildSearchUrl($srchUrl, $this->id); // id
        $this->buildSearchUrl($srchUrl, $this->level); // level
        $this->buildSearchUrl($srchUrl, $this->menu_group); // menu_group
        $this->buildSearchUrl($srchUrl, $this->menu_description); // menu_description
        $this->buildSearchUrl($srchUrl, $this->table_name); // table_name
        $this->buildSearchUrl($srchUrl, $this->table_caption); // table_caption
        $this->buildSearchUrl($srchUrl, $this->field_name); // field_name
        $this->buildSearchUrl($srchUrl, $this->field_caption); // field_caption
        $this->buildSearchUrl($srchUrl, $this->field_description); // field_description
        $this->buildSearchUrl($srchUrl, $this->table_description); // table_description
        $this->buildSearchUrl($srchUrl, $this->portal_notes); // portal_notes
        $this->buildSearchUrl($srchUrl, $this->_visible); // visible
        $this->buildSearchUrl($srchUrl, $this->operator); // operator
        $this->buildSearchUrl($srchUrl, $this->datetime); // datetime
        if ($srchUrl != "") {
            $srchUrl .= "&";
        }
        $srchUrl .= "cmd=search";
        return $srchUrl;
    }

    // Build search URL
    protected function buildSearchUrl(&$url, $fld, $oprOnly = false)
    {
        global $CurrentForm;
        $wrk = "";
        $fldParm = $fld->Param;
        [
            "value" => $fldVal,
            "operator" => $fldOpr,
            "condition" => $fldCond,
            "value2" => $fldVal2,
            "operator2" => $fldOpr2
        ] = $CurrentForm->getSearchValues($fldParm);
        if (is_array($fldVal)) {
            $fldVal = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal);
        }
        if (is_array($fldVal2)) {
            $fldVal2 = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal2);
        }
        $fldDataType = $fld->DataType;
        $value = ConvertSearchValue($fldVal, $fldOpr, $fld); // For testing if numeric only
        $value2 = ConvertSearchValue($fldVal2, $fldOpr2, $fld); // For testing if numeric only
        $fldOpr = ConvertSearchOperator($fldOpr, $fld, $value);
        $fldOpr2 = ConvertSearchOperator($fldOpr2, $fld, $value2);
        if (in_array($fldOpr, ["BETWEEN", "NOT BETWEEN"])) {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld) && IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal != "" && $fldVal2 != "" && $isValidValue) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&y_" . $fldParm . "=" . urlencode($fldVal2) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            }
        } else {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld);
            if ($fldVal != "" && $isValidValue && IsValidOperator($fldOpr)) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            } elseif (in_array($fldOpr, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr != "" && $oprOnly && IsValidOperator($fldOpr))) {
                $wrk = "z_" . $fldParm . "=" . urlencode($fldOpr);
            }
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal2 != "" && $isValidValue && IsValidOperator($fldOpr2)) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "y_" . $fldParm . "=" . urlencode($fldVal2) . "&w_" . $fldParm . "=" . urlencode($fldOpr2);
            } elseif (in_array($fldOpr2, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr2 != "" && $oprOnly && IsValidOperator($fldOpr2))) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "w_" . $fldParm . "=" . urlencode($fldOpr2);
            }
        }
        if ($wrk != "") {
            if ($url != "") {
                $url .= "&";
            }
            $url .= $wrk;
        }
    }

    // Load search values for validation
    protected function loadSearchValues()
    {
        // Load search values
        $hasValue = false;

        // id
        if ($this->id->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // level
        if ($this->level->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // menu_group
        if ($this->menu_group->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // menu_description
        if ($this->menu_description->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // table_name
        if ($this->table_name->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // table_caption
        if ($this->table_caption->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // field_name
        if ($this->field_name->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // field_caption
        if ($this->field_caption->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // field_description
        if ($this->field_description->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // table_description
        if ($this->table_description->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // portal_notes
        if ($this->portal_notes->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // visible
        if ($this->_visible->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // operator
        if ($this->operator->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // datetime
        if ($this->datetime->AdvancedSearch->get()) {
            $hasValue = true;
        }
        return $hasValue;
    }

    // Render row values based on field settings
    public function renderRow()
    {
        global $Security, $Language, $CurrentLanguage;

        // Initialize URLs

        // Call Row_Rendering event
        $this->rowRendering();

        // Common render codes for all row types

        // id
        $this->id->RowCssClass = "row";

        // level
        $this->level->RowCssClass = "row";

        // menu_group
        $this->menu_group->RowCssClass = "row";

        // menu_description
        $this->menu_description->RowCssClass = "row";

        // table_name
        $this->table_name->RowCssClass = "row";

        // table_caption
        $this->table_caption->RowCssClass = "row";

        // field_name
        $this->field_name->RowCssClass = "row";

        // field_caption
        $this->field_caption->RowCssClass = "row";

        // field_description
        $this->field_description->RowCssClass = "row";

        // table_description
        $this->table_description->RowCssClass = "row";

        // system_notes
        $this->system_notes->RowCssClass = "row";

        // portal_notes
        $this->portal_notes->RowCssClass = "row";

        // admin_notes
        $this->admin_notes->RowCssClass = "row";

        // visible
        $this->_visible->RowCssClass = "row";

        // operator
        $this->operator->RowCssClass = "row";

        // datetime
        $this->datetime->RowCssClass = "row";

        // View row
        if ($this->RowType == RowType::VIEW) {
            // id
            $this->id->ViewValue = $this->id->CurrentValue;
            $this->id->ViewValue = FormatNumber($this->id->ViewValue, $this->id->formatPattern());

            // level
            if (strval($this->level->CurrentValue) != "") {
                $this->level->ViewValue = $this->level->optionCaption($this->level->CurrentValue);
            } else {
                $this->level->ViewValue = null;
            }

            // menu_group
            $this->menu_group->ViewValue = $this->menu_group->CurrentValue;

            // menu_description
            $this->menu_description->ViewValue = $this->menu_description->CurrentValue;

            // table_name
            $this->table_name->ViewValue = $this->table_name->CurrentValue;

            // table_caption
            $this->table_caption->ViewValue = $this->table_caption->CurrentValue;

            // field_name
            $this->field_name->ViewValue = $this->field_name->CurrentValue;

            // field_caption
            $this->field_caption->ViewValue = $this->field_caption->CurrentValue;

            // field_description
            $this->field_description->ViewValue = $this->field_description->CurrentValue;

            // table_description
            $this->table_description->ViewValue = $this->table_description->CurrentValue;

            // portal_notes
            $this->portal_notes->ViewValue = $this->portal_notes->CurrentValue;

            // visible
            if (strval($this->_visible->CurrentValue) != "") {
                $this->_visible->ViewValue = $this->_visible->optionCaption($this->_visible->CurrentValue);
            } else {
                $this->_visible->ViewValue = null;
            }

            // operator
            $curVal = strval($this->operator->CurrentValue);
            if ($curVal != "") {
                $this->operator->ViewValue = $this->operator->lookupCacheOption($curVal);
                if ($this->operator->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->operator->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->operator->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                    $lookupFilter = $this->operator->getSelectFilter($this); // PHP
                    $sqlWrk = $this->operator->Lookup->getSql(false, $filterWrk, $lookupFilter, $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->operator->Lookup->renderViewRow($rswrk[0]);
                        $this->operator->ViewValue = $this->operator->displayValue($arwrk);
                    } else {
                        $this->operator->ViewValue = FormatNumber($this->operator->CurrentValue, $this->operator->formatPattern());
                    }
                }
            } else {
                $this->operator->ViewValue = null;
            }

            // datetime
            $this->datetime->ViewValue = $this->datetime->CurrentValue;
            $this->datetime->ViewValue = FormatDateTime($this->datetime->ViewValue, $this->datetime->formatPattern());

            // id
            $this->id->HrefValue = "";
            $this->id->TooltipValue = "";

            // level
            $this->level->HrefValue = "";
            $this->level->TooltipValue = "";

            // menu_group
            $this->menu_group->HrefValue = "";
            $this->menu_group->TooltipValue = "";

            // menu_description
            $this->menu_description->HrefValue = "";
            $this->menu_description->TooltipValue = "";

            // table_name
            $this->table_name->HrefValue = "";
            $this->table_name->TooltipValue = "";

            // table_caption
            $this->table_caption->HrefValue = "";
            $this->table_caption->TooltipValue = "";

            // field_name
            $this->field_name->HrefValue = "";
            $this->field_name->TooltipValue = "";

            // field_caption
            $this->field_caption->HrefValue = "";
            $this->field_caption->TooltipValue = "";

            // field_description
            $this->field_description->HrefValue = "";
            $this->field_description->TooltipValue = "";

            // table_description
            $this->table_description->HrefValue = "";
            $this->table_description->TooltipValue = "";

            // portal_notes
            $this->portal_notes->HrefValue = "";
            $this->portal_notes->TooltipValue = "";

            // visible
            $this->_visible->HrefValue = "";
            $this->_visible->TooltipValue = "";

            // operator
            $this->operator->HrefValue = "";
            $this->operator->TooltipValue = "";

            // datetime
            $this->datetime->HrefValue = "";
            $this->datetime->TooltipValue = "";
        } elseif ($this->RowType == RowType::SEARCH) {
            // id
            $this->id->setupEditAttributes();
            $this->id->EditValue = $this->id->AdvancedSearch->SearchValue;
            $this->id->PlaceHolder = RemoveHtml($this->id->caption());

            // level
            $this->level->setupEditAttributes();
            $this->level->EditValue = $this->level->options(true);
            $this->level->PlaceHolder = RemoveHtml($this->level->caption());

            // menu_group
            $this->menu_group->setupEditAttributes();
            if (!$this->menu_group->Raw) {
                $this->menu_group->AdvancedSearch->SearchValue = HtmlDecode($this->menu_group->AdvancedSearch->SearchValue);
            }
            $this->menu_group->EditValue = HtmlEncode($this->menu_group->AdvancedSearch->SearchValue);
            $this->menu_group->PlaceHolder = RemoveHtml($this->menu_group->caption());
            $this->menu_group->setupEditAttributes();
            if (!$this->menu_group->Raw) {
                $this->menu_group->AdvancedSearch->SearchValue2 = HtmlDecode($this->menu_group->AdvancedSearch->SearchValue2);
            }
            $this->menu_group->EditValue2 = HtmlEncode($this->menu_group->AdvancedSearch->SearchValue2);
            $this->menu_group->PlaceHolder = RemoveHtml($this->menu_group->caption());

            // menu_description
            $this->menu_description->setupEditAttributes();
            $this->menu_description->EditValue = HtmlEncode($this->menu_description->AdvancedSearch->SearchValue);
            $this->menu_description->PlaceHolder = RemoveHtml($this->menu_description->caption());
            $this->menu_description->setupEditAttributes();
            $this->menu_description->EditValue2 = HtmlEncode($this->menu_description->AdvancedSearch->SearchValue2);
            $this->menu_description->PlaceHolder = RemoveHtml($this->menu_description->caption());

            // table_name
            $this->table_name->setupEditAttributes();
            if (!$this->table_name->Raw) {
                $this->table_name->AdvancedSearch->SearchValue = HtmlDecode($this->table_name->AdvancedSearch->SearchValue);
            }
            $this->table_name->EditValue = HtmlEncode($this->table_name->AdvancedSearch->SearchValue);
            $this->table_name->PlaceHolder = RemoveHtml($this->table_name->caption());
            $this->table_name->setupEditAttributes();
            if (!$this->table_name->Raw) {
                $this->table_name->AdvancedSearch->SearchValue2 = HtmlDecode($this->table_name->AdvancedSearch->SearchValue2);
            }
            $this->table_name->EditValue2 = HtmlEncode($this->table_name->AdvancedSearch->SearchValue2);
            $this->table_name->PlaceHolder = RemoveHtml($this->table_name->caption());

            // table_caption
            $this->table_caption->setupEditAttributes();
            if (!$this->table_caption->Raw) {
                $this->table_caption->AdvancedSearch->SearchValue = HtmlDecode($this->table_caption->AdvancedSearch->SearchValue);
            }
            $this->table_caption->EditValue = HtmlEncode($this->table_caption->AdvancedSearch->SearchValue);
            $this->table_caption->PlaceHolder = RemoveHtml($this->table_caption->caption());
            $this->table_caption->setupEditAttributes();
            if (!$this->table_caption->Raw) {
                $this->table_caption->AdvancedSearch->SearchValue2 = HtmlDecode($this->table_caption->AdvancedSearch->SearchValue2);
            }
            $this->table_caption->EditValue2 = HtmlEncode($this->table_caption->AdvancedSearch->SearchValue2);
            $this->table_caption->PlaceHolder = RemoveHtml($this->table_caption->caption());

            // field_name
            $this->field_name->setupEditAttributes();
            if (!$this->field_name->Raw) {
                $this->field_name->AdvancedSearch->SearchValue = HtmlDecode($this->field_name->AdvancedSearch->SearchValue);
            }
            $this->field_name->EditValue = HtmlEncode($this->field_name->AdvancedSearch->SearchValue);
            $this->field_name->PlaceHolder = RemoveHtml($this->field_name->caption());
            $this->field_name->setupEditAttributes();
            if (!$this->field_name->Raw) {
                $this->field_name->AdvancedSearch->SearchValue2 = HtmlDecode($this->field_name->AdvancedSearch->SearchValue2);
            }
            $this->field_name->EditValue2 = HtmlEncode($this->field_name->AdvancedSearch->SearchValue2);
            $this->field_name->PlaceHolder = RemoveHtml($this->field_name->caption());

            // field_caption
            $this->field_caption->setupEditAttributes();
            if (!$this->field_caption->Raw) {
                $this->field_caption->AdvancedSearch->SearchValue = HtmlDecode($this->field_caption->AdvancedSearch->SearchValue);
            }
            $this->field_caption->EditValue = HtmlEncode($this->field_caption->AdvancedSearch->SearchValue);
            $this->field_caption->PlaceHolder = RemoveHtml($this->field_caption->caption());
            $this->field_caption->setupEditAttributes();
            if (!$this->field_caption->Raw) {
                $this->field_caption->AdvancedSearch->SearchValue2 = HtmlDecode($this->field_caption->AdvancedSearch->SearchValue2);
            }
            $this->field_caption->EditValue2 = HtmlEncode($this->field_caption->AdvancedSearch->SearchValue2);
            $this->field_caption->PlaceHolder = RemoveHtml($this->field_caption->caption());

            // field_description
            $this->field_description->setupEditAttributes();
            $this->field_description->EditValue = HtmlEncode($this->field_description->AdvancedSearch->SearchValue);
            $this->field_description->PlaceHolder = RemoveHtml($this->field_description->caption());
            $this->field_description->setupEditAttributes();
            $this->field_description->EditValue2 = HtmlEncode($this->field_description->AdvancedSearch->SearchValue2);
            $this->field_description->PlaceHolder = RemoveHtml($this->field_description->caption());

            // table_description
            $this->table_description->setupEditAttributes();
            $this->table_description->EditValue = HtmlEncode($this->table_description->AdvancedSearch->SearchValue);
            $this->table_description->PlaceHolder = RemoveHtml($this->table_description->caption());

            // portal_notes
            $this->portal_notes->setupEditAttributes();
            $this->portal_notes->EditValue = HtmlEncode($this->portal_notes->AdvancedSearch->SearchValue);
            $this->portal_notes->PlaceHolder = RemoveHtml($this->portal_notes->caption());
            $this->portal_notes->setupEditAttributes();
            $this->portal_notes->EditValue2 = HtmlEncode($this->portal_notes->AdvancedSearch->SearchValue2);
            $this->portal_notes->PlaceHolder = RemoveHtml($this->portal_notes->caption());

            // visible
            $this->_visible->setupEditAttributes();
            $this->_visible->EditValue = $this->_visible->options(true);
            $this->_visible->PlaceHolder = RemoveHtml($this->_visible->caption());

            // operator
            $this->operator->setupEditAttributes();
            $curVal = trim(strval($this->operator->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->operator->AdvancedSearch->ViewValue = $this->operator->lookupCacheOption($curVal);
            } else {
                $this->operator->AdvancedSearch->ViewValue = $this->operator->Lookup !== null && is_array($this->operator->lookupOptions()) && count($this->operator->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->operator->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->operator->EditValue = array_values($this->operator->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->operator->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $this->operator->AdvancedSearch->SearchValue, $this->operator->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                }
                $lookupFilter = $this->operator->getSelectFilter($this); // PHP
                $sqlWrk = $this->operator->Lookup->getSql(true, $filterWrk, $lookupFilter, $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->operator->EditValue = $arwrk;
            }
            $this->operator->PlaceHolder = RemoveHtml($this->operator->caption());

            // datetime
            $this->datetime->setupEditAttributes();
            $this->datetime->EditValue = HtmlEncode(FormatDateTime(UnFormatDateTime($this->datetime->AdvancedSearch->SearchValue, $this->datetime->formatPattern()), $this->datetime->formatPattern()));
            $this->datetime->PlaceHolder = RemoveHtml($this->datetime->caption());
        }
        if ($this->RowType == RowType::ADD || $this->RowType == RowType::EDIT || $this->RowType == RowType::SEARCH) { // Add/Edit/Search row
            $this->setupFieldTitles();
        }

        // Call Row Rendered event
        if ($this->RowType != RowType::AGGREGATEINIT) {
            $this->rowRendered();
        }
    }

    // Validate search
    protected function validateSearch()
    {
        // Check if validation required
        if (!Config("SERVER_VALIDATE")) {
            return true;
        }
        if (!CheckInteger($this->id->AdvancedSearch->SearchValue)) {
            $this->id->addErrorMessage($this->id->getErrorMessage(false));
        }
        if (!CheckDate($this->datetime->AdvancedSearch->SearchValue, $this->datetime->formatPattern())) {
            $this->datetime->addErrorMessage($this->datetime->getErrorMessage(false));
        }

        // Return validate result
        $validateSearch = !$this->hasInvalidFields();

        // Call Form_CustomValidate event
        $formCustomError = "";
        $validateSearch = $validateSearch && $this->formCustomValidate($formCustomError);
        if ($formCustomError != "") {
            $this->setFailureMessage($formCustomError);
        }
        return $validateSearch;
    }

    // Load advanced search
    public function loadAdvancedSearch()
    {
        $this->id->AdvancedSearch->load();
        $this->level->AdvancedSearch->load();
        $this->menu_group->AdvancedSearch->load();
        $this->menu_description->AdvancedSearch->load();
        $this->table_name->AdvancedSearch->load();
        $this->table_caption->AdvancedSearch->load();
        $this->field_name->AdvancedSearch->load();
        $this->field_caption->AdvancedSearch->load();
        $this->field_description->AdvancedSearch->load();
        $this->table_description->AdvancedSearch->load();
        $this->portal_notes->AdvancedSearch->load();
        $this->_visible->AdvancedSearch->load();
        $this->operator->AdvancedSearch->load();
        $this->datetime->AdvancedSearch->load();
    }

    // Set up Breadcrumb
    protected function setupBreadcrumb()
    {
        global $Breadcrumb, $Language;
        $Breadcrumb = new Breadcrumb("db2");
        $url = CurrentUrl();
        $Breadcrumb->add("list", $this->TableVar, $this->addMasterUrl("helplist"), "", $this->TableVar, true);
        $pageId = "search";
        $Breadcrumb->add("search", $pageId, $url);
    }

    // Setup lookup options
    public function setupLookupOptions($fld)
    {
        if ($fld->Lookup && $fld->Lookup->Options === null) {
            // Get default connection and filter
            $conn = $this->getConnection();
            $lookupFilter = "";

            // No need to check any more
            $fld->Lookup->Options = [];

            // Set up lookup SQL and connection
            switch ($fld->FieldVar) {
                case "x_level":
                    break;
                case "x__visible":
                    break;
                case "x_operator":
                    $lookupFilter = $fld->getSelectFilter(); // PHP
                    break;
                default:
                    $lookupFilter = "";
                    break;
            }

            // Always call to Lookup->getSql so that user can setup Lookup->Options in Lookup_Selecting server event
            $sql = $fld->Lookup->getSql(false, "", $lookupFilter, $this);

            // Set up lookup cache
            if (!$fld->hasLookupOptions() && $fld->UseLookupCache && $sql != "" && count($fld->Lookup->Options) == 0 && count($fld->Lookup->FilterFields) == 0) {
                $totalCnt = $this->getRecordCount($sql, $conn);
                if ($totalCnt > $fld->LookupCacheCount) { // Total count > cache count, do not cache
                    return;
                }
                $rows = $conn->executeQuery($sql)->fetchAll();
                $ar = [];
                foreach ($rows as $row) {
                    $row = $fld->Lookup->renderViewRow($row, Container($fld->Lookup->LinkTable));
                    $key = $row["lf"];
                    if (IsFloatType($fld->Type)) { // Handle float field
                        $key = (float)$key;
                    }
                    $ar[strval($key)] = $row;
                }
                $fld->Lookup->Options = $ar;
            }
        }
    }

    // Page Load event
    public function pageLoad()
    {
        //Log("Page Load");
    }

    // Page Unload event
    public function pageUnload()
    {
        //Log("Page Unload");
    }

    // Page Redirecting event
    public function pageRedirecting(&$url)
    {
        // Example:
        //$url = "your URL";
    }

    // Message Showing event
    // $type = ''|'success'|'failure'|'warning'
    public function messageShowing(&$msg, $type)
    {
        if ($type == "success") {
            //$msg = "your success message";
        } elseif ($type == "failure") {
            //$msg = "your failure message";
        } elseif ($type == "warning") {
            //$msg = "your warning message";
        } else {
            //$msg = "your message";
        }
    }

    // Page Render event
    public function pageRender() {
    	//echo "Page Render";
    if ((CurrentUserID() == -1) && ($_SERVER["HTTP_HOST"] == 'fibre.intsys.in'))
    {
    	$this->table_name->Visible = True;
    	$this->field_name->Visible = True;
    	$this->field_caption->Visible = True;
    	$this->field_description->Visible = True;
    	$this->table_description->Visible = True;
    	$this->system_notes->Visible = True;
    	$this->_visible->Visible = True;
    }
    else
    {
    	$this->table_name->Visible = False;
    	$this->field_name->Visible = False;
    	$this->field_caption->Visible = False;
    	$this->field_description->Visible = False;
    	$this->table_description->Visible = False;
    	$this->system_notes->Visible = False;
    	$this->_visible->Visible = False;
    }	
    }

    // Page Data Rendering event
    public function pageDataRendering(&$header)
    {
        // Example:
        //$header = "your header";
    }

    // Page Data Rendered event
    public function pageDataRendered(&$footer)
    {
        // Example:
        //$footer = "your footer";
    }

    // Page Breaking event
    public function pageBreaking(&$break, &$content)
    {
        // Example:
        //$break = false; // Skip page break, or
        //$content = "<div style=\"break-after:page;\"></div>"; // Modify page break content
    }

    // Form Custom Validate event
    public function formCustomValidate(&$customError)
    {
        // Return error message in $customError
        return true;
    }
}
