<?php

namespace PHPMaker2024\fibre;

use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use Psr\Http\Message\ServerRequestInterface as Request;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Container\ContainerInterface;
use Slim\Routing\RouteCollectorProxy;
use Slim\App;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Closure;

/**
 * Page class
 */
class ApcSearch extends Apc
{
    use MessagesTrait;

    // Page ID
    public $PageID = "search";

    // Project ID
    public $ProjectID = PROJECT_ID;

    // Page object name
    public $PageObjName = "ApcSearch";

    // View file path
    public $View = null;

    // Title
    public $Title = null; // Title for <title> tag

    // Rendering View
    public $RenderingView = false;

    // CSS class/style
    public $CurrentPageName = "apcsearch";

    // Page headings
    public $Heading = "";
    public $Subheading = "";
    public $PageHeader;
    public $PageFooter;

    // Page layout
    public $UseLayout = true;

    // Page terminated
    private $terminated = false;

    // Page heading
    public function pageHeading()
    {
        global $Language;
        if ($this->Heading != "") {
            return $this->Heading;
        }
        if (method_exists($this, "tableCaption")) {
            return $this->tableCaption();
        }
        return "";
    }

    // Page subheading
    public function pageSubheading()
    {
        global $Language;
        if ($this->Subheading != "") {
            return $this->Subheading;
        }
        if ($this->TableName) {
            return $Language->phrase($this->PageID);
        }
        return "";
    }

    // Page name
    public function pageName()
    {
        return CurrentPageName();
    }

    // Page URL
    public function pageUrl($withArgs = true)
    {
        $route = GetRoute();
        $args = RemoveXss($route->getArguments());
        if (!$withArgs) {
            foreach ($args as $key => &$val) {
                $val = "";
            }
            unset($val);
        }
        return rtrim(UrlFor($route->getName(), $args), "/") . "?";
    }

    // Show Page Header
    public function showPageHeader()
    {
        $header = $this->PageHeader;
        $this->pageDataRendering($header);
        if ($header != "") { // Header exists, display
            echo '<div id="ew-page-header">' . $header . '</div>';
        }
    }

    // Show Page Footer
    public function showPageFooter()
    {
        $footer = $this->PageFooter;
        $this->pageDataRendered($footer);
        if ($footer != "") { // Footer exists, display
            echo '<div id="ew-page-footer">' . $footer . '</div>';
        }
    }

    // Set field visibility
    public function setVisibility()
    {
        $this->id->Visible = false;
        $this->apcid->setVisibility();
        $this->grp->setVisibility();
        $this->code->setVisibility();
        $this->description->setVisibility();
        $this->cron->setVisibility();
        $this->next_cycle->setVisibility();
        $this->delay->setVisibility();
        $this->delay_method->setVisibility();
        $this->last_updated->setVisibility();
        $this->user_level->setVisibility();
        $this->_visible->setVisibility();
        $this->operator->setVisibility();
        $this->datetime->setVisibility();
        $this->status->setVisibility();
        $this->order_by->setVisibility();
        $this->url->setVisibility();
    }

    // Constructor
    public function __construct()
    {
        parent::__construct();
        global $Language, $DashboardReport, $DebugTimer, $UserTable;
        $this->TableVar = 'apc';
        $this->TableName = 'apc';

        // Table CSS class
        $this->TableClass = "table table-striped table-bordered table-hover table-sm ew-desktop-table ew-search-table";

        // Initialize
        $GLOBALS["Page"] = &$this;

        // Language object
        $Language = Container("app.language");

        // Table object (apc)
        if (!isset($GLOBALS["apc"]) || $GLOBALS["apc"]::class == PROJECT_NAMESPACE . "apc") {
            $GLOBALS["apc"] = &$this;
        }

        // Table name (for backward compatibility only)
        if (!defined(PROJECT_NAMESPACE . "TABLE_NAME")) {
            define(PROJECT_NAMESPACE . "TABLE_NAME", 'apc');
        }

        // Start timer
        $DebugTimer = Container("debug.timer");

        // Debug message
        LoadDebugMessage();

        // Open connection
        $GLOBALS["Conn"] ??= $this->getConnection();

        // User table object
        $UserTable = Container("usertable");
    }

    // Get content from stream
    public function getContents(): string
    {
        global $Response;
        return $Response?->getBody() ?? ob_get_clean();
    }

    // Is lookup
    public function isLookup()
    {
        return SameText(Route(0), Config("API_LOOKUP_ACTION"));
    }

    // Is AutoFill
    public function isAutoFill()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autofill");
    }

    // Is AutoSuggest
    public function isAutoSuggest()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autosuggest");
    }

    // Is modal lookup
    public function isModalLookup()
    {
        return $this->isLookup() && SameText(Post("ajax"), "modal");
    }

    // Is terminated
    public function isTerminated()
    {
        return $this->terminated;
    }

    /**
     * Terminate page
     *
     * @param string $url URL for direction
     * @return void
     */
    public function terminate($url = "")
    {
        if ($this->terminated) {
            return;
        }
        global $TempImages, $DashboardReport, $Response;

        // Page is terminated
        $this->terminated = true;

        // Page Unload event
        if (method_exists($this, "pageUnload")) {
            $this->pageUnload();
        }
        DispatchEvent(new PageUnloadedEvent($this), PageUnloadedEvent::NAME);
        if (!IsApi() && method_exists($this, "pageRedirecting")) {
            $this->pageRedirecting($url);
        }

        // Close connection
        CloseConnections();

        // Return for API
        if (IsApi()) {
            $res = $url === true;
            if (!$res) { // Show response for API
                $ar = array_merge($this->getMessages(), $url ? ["url" => GetUrl($url)] : []);
                WriteJson($ar);
            }
            $this->clearMessages(); // Clear messages for API request
            return;
        } else { // Check if response is JSON
            if (WithJsonResponse()) { // With JSON response
                $this->clearMessages();
                return;
            }
        }

        // Go to URL if specified
        if ($url != "") {
            if (!Config("DEBUG") && ob_get_length()) {
                ob_end_clean();
            }

            // Handle modal response
            if ($this->IsModal) { // Show as modal
                $pageName = GetPageName($url);
                $result = ["url" => GetUrl($url), "modal" => "1"];  // Assume return to modal for simplicity
                if (!SameString($pageName, GetPageName($this->getListUrl()))) { // Not List page
                    $result["caption"] = $this->getModalCaption($pageName);
                    $result["view"] = SameString($pageName, "apcview"); // If View page, no primary button
                } else { // List page
                    $result["error"] = $this->getFailureMessage(); // List page should not be shown as modal => error
                    $this->clearFailureMessage();
                }
                WriteJson($result);
            } else {
                SaveDebugMessage();
                Redirect(GetUrl($url));
            }
        }
        return; // Return to controller
    }

    // Get records from result set
    protected function getRecordsFromRecordset($rs, $current = false)
    {
        $rows = [];
        if (is_object($rs)) { // Result set
            while ($row = $rs->fetch()) {
                $this->loadRowValues($row); // Set up DbValue/CurrentValue
                $row = $this->getRecordFromArray($row);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        } elseif (is_array($rs)) {
            foreach ($rs as $ar) {
                $row = $this->getRecordFromArray($ar);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        }
        return $rows;
    }

    // Get record from array
    protected function getRecordFromArray($ar)
    {
        $row = [];
        if (is_array($ar)) {
            foreach ($ar as $fldname => $val) {
                if (array_key_exists($fldname, $this->Fields) && ($this->Fields[$fldname]->Visible || $this->Fields[$fldname]->IsPrimaryKey)) { // Primary key or Visible
                    $fld = &$this->Fields[$fldname];
                    if ($fld->HtmlTag == "FILE") { // Upload field
                        if (EmptyValue($val)) {
                            $row[$fldname] = null;
                        } else {
                            if ($fld->DataType == DataType::BLOB) {
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . $fld->Param . "/" . rawurlencode($this->getRecordKeyValue($ar))));
                                $row[$fldname] = ["type" => ContentType($val), "url" => $url, "name" => $fld->Param . ContentExtension($val)];
                            } elseif (!$fld->UploadMultiple || !ContainsString($val, Config("MULTIPLE_UPLOAD_SEPARATOR"))) { // Single file
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $val)));
                                $row[$fldname] = ["type" => MimeContentType($val), "url" => $url, "name" => $val];
                            } else { // Multiple files
                                $files = explode(Config("MULTIPLE_UPLOAD_SEPARATOR"), $val);
                                $ar = [];
                                foreach ($files as $file) {
                                    $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                        "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $file)));
                                    if (!EmptyValue($file)) {
                                        $ar[] = ["type" => MimeContentType($file), "url" => $url, "name" => $file];
                                    }
                                }
                                $row[$fldname] = $ar;
                            }
                        }
                    } else {
                        $row[$fldname] = $val;
                    }
                }
            }
        }
        return $row;
    }

    // Get record key value from array
    protected function getRecordKeyValue($ar)
    {
        $key = "";
        if (is_array($ar)) {
            $key .= @$ar['apcid'];
        }
        return $key;
    }

    /**
     * Hide fields for add/edit
     *
     * @return void
     */
    protected function hideFieldsForAddEdit()
    {
    }

    // Lookup data
    public function lookup(array $req = [], bool $response = true)
    {
        global $Language, $Security;

        // Get lookup object
        $fieldName = $req["field"] ?? null;
        if (!$fieldName) {
            return [];
        }
        $fld = $this->Fields[$fieldName];
        $lookup = $fld->Lookup;
        $name = $req["name"] ?? "";
        if (ContainsString($name, "query_builder_rule")) {
            $lookup->FilterFields = []; // Skip parent fields if any
        }

        // Get lookup parameters
        $lookupType = $req["ajax"] ?? "unknown";
        $pageSize = -1;
        $offset = -1;
        $searchValue = "";
        if (SameText($lookupType, "modal") || SameText($lookupType, "filter")) {
            $searchValue = $req["q"] ?? $req["sv"] ?? "";
            $pageSize = $req["n"] ?? $req["recperpage"] ?? 10;
        } elseif (SameText($lookupType, "autosuggest")) {
            $searchValue = $req["q"] ?? "";
            $pageSize = $req["n"] ?? -1;
            $pageSize = is_numeric($pageSize) ? (int)$pageSize : -1;
            if ($pageSize <= 0) {
                $pageSize = Config("AUTO_SUGGEST_MAX_ENTRIES");
            }
        }
        $start = $req["start"] ?? -1;
        $start = is_numeric($start) ? (int)$start : -1;
        $page = $req["page"] ?? -1;
        $page = is_numeric($page) ? (int)$page : -1;
        $offset = $start >= 0 ? $start : ($page > 0 && $pageSize > 0 ? ($page - 1) * $pageSize : 0);
        $userSelect = Decrypt($req["s"] ?? "");
        $userFilter = Decrypt($req["f"] ?? "");
        $userOrderBy = Decrypt($req["o"] ?? "");
        $keys = $req["keys"] ?? null;
        $lookup->LookupType = $lookupType; // Lookup type
        $lookup->FilterValues = []; // Clear filter values first
        if ($keys !== null) { // Selected records from modal
            if (is_array($keys)) {
                $keys = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $keys);
            }
            $lookup->FilterFields = []; // Skip parent fields if any
            $lookup->FilterValues[] = $keys; // Lookup values
            $pageSize = -1; // Show all records
        } else { // Lookup values
            $lookup->FilterValues[] = $req["v0"] ?? $req["lookupValue"] ?? "";
        }
        $cnt = is_array($lookup->FilterFields) ? count($lookup->FilterFields) : 0;
        for ($i = 1; $i <= $cnt; $i++) {
            $lookup->FilterValues[] = $req["v" . $i] ?? "";
        }
        $lookup->SearchValue = $searchValue;
        $lookup->PageSize = $pageSize;
        $lookup->Offset = $offset;
        if ($userSelect != "") {
            $lookup->UserSelect = $userSelect;
        }
        if ($userFilter != "") {
            $lookup->UserFilter = $userFilter;
        }
        if ($userOrderBy != "") {
            $lookup->UserOrderBy = $userOrderBy;
        }
        return $lookup->toJson($this, $response); // Use settings from current page
    }
    public $FormClassName = "ew-form ew-search-form";
    public $IsModal = false;
    public $IsMobileOrModal = false;

    /**
     * Page run
     *
     * @return void
     */
    public function run()
    {
        global $ExportType, $Language, $Security, $CurrentForm, $SkipHeaderFooter;

        // Is modal
        $this->IsModal = ConvertToBool(Param("modal"));
        $this->UseLayout = $this->UseLayout && !$this->IsModal;

        // Use layout
        $this->UseLayout = $this->UseLayout && ConvertToBool(Param(Config("PAGE_LAYOUT"), true));

        // View
        $this->View = Get(Config("VIEW"));

        // Load user profile
        if (IsLoggedIn()) {
            Profile()->setUserName(CurrentUserName())->loadFromStorage();
        }

        // Create form object
        $CurrentForm = new HttpForm();
        $this->CurrentAction = Param("action"); // Set up current action
        $this->setVisibility();

        // Set lookup cache
        if (!in_array($this->PageID, Config("LOOKUP_CACHE_PAGE_IDS"))) {
            $this->setUseLookupCache(false);
        }

        // Global Page Loading event (in userfn*.php)
        DispatchEvent(new PageLoadingEvent($this), PageLoadingEvent::NAME);

        // Page Load event
        if (method_exists($this, "pageLoad")) {
            $this->pageLoad();
        }

        // Hide fields for add/edit
        if (!$this->UseAjaxActions) {
            $this->hideFieldsForAddEdit();
        }
        // Use inline delete
        if ($this->UseAjaxActions) {
            $this->InlineDelete = true;
        }

        // Set up lookup cache
        $this->setupLookupOptions($this->grp);
        $this->setupLookupOptions($this->cron);
        $this->setupLookupOptions($this->delay_method);
        $this->setupLookupOptions($this->user_level);
        $this->setupLookupOptions($this->_visible);
        $this->setupLookupOptions($this->operator);
        $this->setupLookupOptions($this->status);

        // Set up Breadcrumb
        $this->setupBreadcrumb();

        // Check modal
        if ($this->IsModal) {
            $SkipHeaderFooter = true;
        }
        $this->IsMobileOrModal = IsMobile() || $this->IsModal;

        // Get action
        $this->CurrentAction = Post("action");
        if ($this->isSearch()) {
            // Build search string for advanced search, remove blank field
            $this->loadSearchValues(); // Get search values
            $srchStr = $this->validateSearch() ? $this->buildAdvancedSearch() : "";
            if ($srchStr != "") {
                $srchStr = "apclist" . "?" . $srchStr;
                // Do not return Json for UseAjaxActions
                if ($this->IsModal && $this->UseAjaxActions) {
                    $this->IsModal = false;
                }
                $this->terminate($srchStr); // Go to list page
                return;
            }
        }

        // Restore search settings from Session
        if (!$this->hasInvalidFields()) {
            $this->loadAdvancedSearch();
        }

        // Render row for search
        $this->RowType = RowType::SEARCH;
        $this->resetAttributes();
        $this->renderRow();

        // Set LoginStatus / Page_Rendering / Page_Render
        if (!IsApi() && !$this->isTerminated()) {
            // Setup login status
            SetupLoginStatus();

            // Pass login status to client side
            SetClientVar("login", LoginStatus());

            // Global Page Rendering event (in userfn*.php)
            DispatchEvent(new PageRenderingEvent($this), PageRenderingEvent::NAME);

            // Page Render event
            if (method_exists($this, "pageRender")) {
                $this->pageRender();
            }

            // Render search option
            if (method_exists($this, "renderSearchOptions")) {
                $this->renderSearchOptions();
            }
        }
    }

    // Build advanced search
    protected function buildAdvancedSearch()
    {
        $srchUrl = "";
        $this->buildSearchUrl($srchUrl, $this->apcid); // apcid
        $this->buildSearchUrl($srchUrl, $this->grp); // grp
        $this->buildSearchUrl($srchUrl, $this->code); // code
        $this->buildSearchUrl($srchUrl, $this->description); // description
        $this->buildSearchUrl($srchUrl, $this->cron); // cron
        $this->buildSearchUrl($srchUrl, $this->next_cycle); // next_cycle
        $this->buildSearchUrl($srchUrl, $this->delay); // delay
        $this->buildSearchUrl($srchUrl, $this->delay_method); // delay_method
        $this->buildSearchUrl($srchUrl, $this->last_updated); // last_updated
        $this->buildSearchUrl($srchUrl, $this->user_level); // user_level
        $this->buildSearchUrl($srchUrl, $this->_visible); // visible
        $this->buildSearchUrl($srchUrl, $this->operator); // operator
        $this->buildSearchUrl($srchUrl, $this->datetime); // datetime
        $this->buildSearchUrl($srchUrl, $this->status); // status
        $this->buildSearchUrl($srchUrl, $this->order_by); // order_by
        $this->buildSearchUrl($srchUrl, $this->url); // url
        if ($srchUrl != "") {
            $srchUrl .= "&";
        }
        $srchUrl .= "cmd=search";
        return $srchUrl;
    }

    // Build search URL
    protected function buildSearchUrl(&$url, $fld, $oprOnly = false)
    {
        global $CurrentForm;
        $wrk = "";
        $fldParm = $fld->Param;
        [
            "value" => $fldVal,
            "operator" => $fldOpr,
            "condition" => $fldCond,
            "value2" => $fldVal2,
            "operator2" => $fldOpr2
        ] = $CurrentForm->getSearchValues($fldParm);
        if (is_array($fldVal)) {
            $fldVal = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal);
        }
        if (is_array($fldVal2)) {
            $fldVal2 = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal2);
        }
        $fldDataType = $fld->DataType;
        $value = ConvertSearchValue($fldVal, $fldOpr, $fld); // For testing if numeric only
        $value2 = ConvertSearchValue($fldVal2, $fldOpr2, $fld); // For testing if numeric only
        $fldOpr = ConvertSearchOperator($fldOpr, $fld, $value);
        $fldOpr2 = ConvertSearchOperator($fldOpr2, $fld, $value2);
        if (in_array($fldOpr, ["BETWEEN", "NOT BETWEEN"])) {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld) && IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal != "" && $fldVal2 != "" && $isValidValue) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&y_" . $fldParm . "=" . urlencode($fldVal2) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            }
        } else {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld);
            if ($fldVal != "" && $isValidValue && IsValidOperator($fldOpr)) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            } elseif (in_array($fldOpr, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr != "" && $oprOnly && IsValidOperator($fldOpr))) {
                $wrk = "z_" . $fldParm . "=" . urlencode($fldOpr);
            }
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal2 != "" && $isValidValue && IsValidOperator($fldOpr2)) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "y_" . $fldParm . "=" . urlencode($fldVal2) . "&w_" . $fldParm . "=" . urlencode($fldOpr2);
            } elseif (in_array($fldOpr2, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr2 != "" && $oprOnly && IsValidOperator($fldOpr2))) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "w_" . $fldParm . "=" . urlencode($fldOpr2);
            }
        }
        if ($wrk != "") {
            if ($url != "") {
                $url .= "&";
            }
            $url .= $wrk;
        }
    }

    // Load search values for validation
    protected function loadSearchValues()
    {
        // Load search values
        $hasValue = false;

        // apcid
        if ($this->apcid->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // grp
        if ($this->grp->AdvancedSearch->get()) {
            $hasValue = true;
        }
        if (is_array($this->grp->AdvancedSearch->SearchValue)) {
            $this->grp->AdvancedSearch->SearchValue = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $this->grp->AdvancedSearch->SearchValue);
        }
        if (is_array($this->grp->AdvancedSearch->SearchValue2)) {
            $this->grp->AdvancedSearch->SearchValue2 = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $this->grp->AdvancedSearch->SearchValue2);
        }

        // code
        if ($this->code->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // description
        if ($this->description->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // cron
        if ($this->cron->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // next_cycle
        if ($this->next_cycle->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // delay
        if ($this->delay->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // delay_method
        if ($this->delay_method->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // last_updated
        if ($this->last_updated->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // user_level
        if ($this->user_level->AdvancedSearch->get()) {
            $hasValue = true;
        }
        if (is_array($this->user_level->AdvancedSearch->SearchValue)) {
            $this->user_level->AdvancedSearch->SearchValue = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $this->user_level->AdvancedSearch->SearchValue);
        }
        if (is_array($this->user_level->AdvancedSearch->SearchValue2)) {
            $this->user_level->AdvancedSearch->SearchValue2 = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $this->user_level->AdvancedSearch->SearchValue2);
        }

        // visible
        if ($this->_visible->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // operator
        if ($this->operator->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // datetime
        if ($this->datetime->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // status
        if ($this->status->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // order_by
        if ($this->order_by->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // url
        if ($this->url->AdvancedSearch->get()) {
            $hasValue = true;
        }
        return $hasValue;
    }

    // Render row values based on field settings
    public function renderRow()
    {
        global $Security, $Language, $CurrentLanguage;

        // Initialize URLs

        // Call Row_Rendering event
        $this->rowRendering();

        // Common render codes for all row types

        // id
        $this->id->RowCssClass = "row";

        // apcid
        $this->apcid->RowCssClass = "row";

        // grp
        $this->grp->RowCssClass = "row";

        // code
        $this->code->RowCssClass = "row";

        // description
        $this->description->RowCssClass = "row";

        // cron
        $this->cron->RowCssClass = "row";

        // next_cycle
        $this->next_cycle->RowCssClass = "row";

        // delay
        $this->delay->RowCssClass = "row";

        // delay_method
        $this->delay_method->RowCssClass = "row";

        // last_updated
        $this->last_updated->RowCssClass = "row";

        // user_level
        $this->user_level->RowCssClass = "row";

        // visible
        $this->_visible->RowCssClass = "row";

        // operator
        $this->operator->RowCssClass = "row";

        // datetime
        $this->datetime->RowCssClass = "row";

        // status
        $this->status->RowCssClass = "row";

        // order_by
        $this->order_by->RowCssClass = "row";

        // url
        $this->url->RowCssClass = "row";

        // View row
        if ($this->RowType == RowType::VIEW) {
            // apcid
            $this->apcid->ViewValue = $this->apcid->CurrentValue;

            // grp
            $curVal = strval($this->grp->CurrentValue);
            if ($curVal != "") {
                $this->grp->ViewValue = $this->grp->lookupCacheOption($curVal);
                if ($this->grp->ViewValue === null) { // Lookup from database
                    $arwrk = explode(Config("MULTIPLE_OPTION_SEPARATOR"), $curVal);
                    $filterWrk = "";
                    foreach ($arwrk as $wrk) {
                        AddFilter($filterWrk, SearchFilter($this->grp->Lookup->getTable()->Fields["id"]->searchExpression(), "=", trim($wrk), $this->grp->Lookup->getTable()->Fields["id"]->searchDataType(), ""), "OR");
                    }
                    $lookupFilter = $this->grp->getSelectFilter($this); // PHP
                    $sqlWrk = $this->grp->Lookup->getSql(false, $filterWrk, $lookupFilter, $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $this->grp->ViewValue = new OptionValues();
                        foreach ($rswrk as $row) {
                            $arwrk = $this->grp->Lookup->renderViewRow($row);
                            $this->grp->ViewValue->add($this->grp->displayValue($arwrk));
                        }
                    } else {
                        $this->grp->ViewValue = $this->grp->CurrentValue;
                    }
                }
            } else {
                $this->grp->ViewValue = null;
            }

            // code
            $this->code->ViewValue = $this->code->CurrentValue;

            // description
            $this->description->ViewValue = $this->description->CurrentValue;

            // cron
            if (strval($this->cron->CurrentValue) != "") {
                $this->cron->ViewValue = $this->cron->optionCaption($this->cron->CurrentValue);
            } else {
                $this->cron->ViewValue = null;
            }

            // next_cycle
            $this->next_cycle->ViewValue = $this->next_cycle->CurrentValue;
            $this->next_cycle->ViewValue = FormatDateTime($this->next_cycle->ViewValue, $this->next_cycle->formatPattern());

            // delay
            $this->delay->ViewValue = $this->delay->CurrentValue;

            // delay_method
            if (strval($this->delay_method->CurrentValue) != "") {
                $this->delay_method->ViewValue = $this->delay_method->optionCaption($this->delay_method->CurrentValue);
            } else {
                $this->delay_method->ViewValue = null;
            }

            // last_updated
            $this->last_updated->ViewValue = $this->last_updated->CurrentValue;

            // user_level
            $curVal = strval($this->user_level->CurrentValue);
            if ($curVal != "") {
                $this->user_level->ViewValue = $this->user_level->lookupCacheOption($curVal);
                if ($this->user_level->ViewValue === null) { // Lookup from database
                    $arwrk = explode(Config("MULTIPLE_OPTION_SEPARATOR"), $curVal);
                    $filterWrk = "";
                    foreach ($arwrk as $wrk) {
                        AddFilter($filterWrk, SearchFilter($this->user_level->Lookup->getTable()->Fields["userlevelid"]->searchExpression(), "=", trim($wrk), $this->user_level->Lookup->getTable()->Fields["userlevelid"]->searchDataType(), ""), "OR");
                    }
                    $sqlWrk = $this->user_level->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $this->user_level->ViewValue = new OptionValues();
                        foreach ($rswrk as $row) {
                            $arwrk = $this->user_level->Lookup->renderViewRow($row);
                            $this->user_level->ViewValue->add($this->user_level->displayValue($arwrk));
                        }
                    } else {
                        $this->user_level->ViewValue = $this->user_level->CurrentValue;
                    }
                }
            } else {
                $this->user_level->ViewValue = null;
            }

            // visible
            if (strval($this->_visible->CurrentValue) != "") {
                $this->_visible->ViewValue = $this->_visible->optionCaption($this->_visible->CurrentValue);
            } else {
                $this->_visible->ViewValue = null;
            }

            // operator
            $curVal = strval($this->operator->CurrentValue);
            if ($curVal != "") {
                $this->operator->ViewValue = $this->operator->lookupCacheOption($curVal);
                if ($this->operator->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->operator->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $curVal, $this->operator->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                    $sqlWrk = $this->operator->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->operator->Lookup->renderViewRow($rswrk[0]);
                        $this->operator->ViewValue = $this->operator->displayValue($arwrk);
                    } else {
                        $this->operator->ViewValue = $this->operator->CurrentValue;
                    }
                }
            } else {
                $this->operator->ViewValue = null;
            }

            // datetime
            $this->datetime->ViewValue = $this->datetime->CurrentValue;
            $this->datetime->ViewValue = FormatDateTime($this->datetime->ViewValue, $this->datetime->formatPattern());

            // status
            if (strval($this->status->CurrentValue) != "") {
                $this->status->ViewValue = $this->status->optionCaption($this->status->CurrentValue);
            } else {
                $this->status->ViewValue = null;
            }

            // order_by
            $this->order_by->ViewValue = $this->order_by->CurrentValue;
            $this->order_by->ViewValue = FormatNumber($this->order_by->ViewValue, $this->order_by->formatPattern());

            // url
            $this->url->ViewValue = $this->url->CurrentValue;

            // apcid
            $this->apcid->HrefValue = "";
            $this->apcid->TooltipValue = "";

            // grp
            $this->grp->HrefValue = "";
            $this->grp->TooltipValue = "";

            // code
            $this->code->HrefValue = "";
            $this->code->TooltipValue = "";

            // description
            if (!EmptyValue($this->id->CurrentValue)) {
                $this->description->HrefValue = $this->description->getLinkPrefix() . $this->id->CurrentValue; // Add prefix/suffix
                $this->description->LinkAttrs["target"] = ""; // Add target
                if ($this->isExport()) {
                    $this->description->HrefValue = FullUrl($this->description->HrefValue, "href");
                }
            } else {
                $this->description->HrefValue = "";
            }
            $this->description->TooltipValue = "";

            // cron
            $this->cron->HrefValue = "";
            $this->cron->TooltipValue = "";

            // next_cycle
            $this->next_cycle->HrefValue = "";
            $this->next_cycle->TooltipValue = "";

            // delay
            $this->delay->HrefValue = "";
            $this->delay->TooltipValue = "";

            // delay_method
            $this->delay_method->HrefValue = "";
            $this->delay_method->TooltipValue = "";

            // last_updated
            $this->last_updated->HrefValue = "";
            $this->last_updated->TooltipValue = "";

            // user_level
            $this->user_level->HrefValue = "";
            $this->user_level->TooltipValue = "";

            // visible
            $this->_visible->HrefValue = "";
            $this->_visible->TooltipValue = "";

            // operator
            $this->operator->HrefValue = "";
            $this->operator->TooltipValue = "";

            // datetime
            $this->datetime->HrefValue = "";
            $this->datetime->TooltipValue = "";

            // status
            $this->status->HrefValue = "";
            $this->status->TooltipValue = "";

            // order_by
            $this->order_by->HrefValue = "";
            $this->order_by->TooltipValue = "";

            // url
            $this->url->HrefValue = "";
            $this->url->TooltipValue = "";
        } elseif ($this->RowType == RowType::SEARCH) {
            // apcid
            $this->apcid->setupEditAttributes();
            $this->apcid->EditValue = $this->apcid->AdvancedSearch->SearchValue;
            $this->apcid->PlaceHolder = RemoveHtml($this->apcid->caption());

            // grp
            $curVal = trim(strval($this->grp->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->grp->AdvancedSearch->ViewValue = $this->grp->lookupCacheOption($curVal);
            } else {
                $this->grp->AdvancedSearch->ViewValue = $this->grp->Lookup !== null && is_array($this->grp->lookupOptions()) && count($this->grp->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->grp->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->grp->EditValue = array_values($this->grp->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $arwrk = explode(Config("MULTIPLE_OPTION_SEPARATOR"), $curVal);
                    $filterWrk = "";
                    foreach ($arwrk as $wrk) {
                        AddFilter($filterWrk, SearchFilter($this->grp->Lookup->getTable()->Fields["id"]->searchExpression(), "=", trim($wrk), $this->grp->Lookup->getTable()->Fields["id"]->searchDataType(), ""), "OR");
                    }
                }
                $lookupFilter = $this->grp->getSelectFilter($this); // PHP
                $sqlWrk = $this->grp->Lookup->getSql(true, $filterWrk, $lookupFilter, $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->grp->EditValue = $arwrk;
            }
            $this->grp->PlaceHolder = RemoveHtml($this->grp->caption());

            // code
            $this->code->setupEditAttributes();
            $this->code->EditValue = HtmlEncode($this->code->AdvancedSearch->SearchValue);
            $this->code->PlaceHolder = RemoveHtml($this->code->caption());

            // description
            $this->description->setupEditAttributes();
            if (!$this->description->Raw) {
                $this->description->AdvancedSearch->SearchValue = HtmlDecode($this->description->AdvancedSearch->SearchValue);
            }
            $this->description->EditValue = HtmlEncode($this->description->AdvancedSearch->SearchValue);
            $this->description->PlaceHolder = RemoveHtml($this->description->caption());

            // cron
            $this->cron->setupEditAttributes();
            $this->cron->EditValue = $this->cron->options(true);
            $this->cron->PlaceHolder = RemoveHtml($this->cron->caption());

            // next_cycle
            $this->next_cycle->setupEditAttributes();
            $this->next_cycle->EditValue = HtmlEncode(FormatDateTime(UnFormatDateTime($this->next_cycle->AdvancedSearch->SearchValue, $this->next_cycle->formatPattern()), $this->next_cycle->formatPattern()));
            $this->next_cycle->PlaceHolder = RemoveHtml($this->next_cycle->caption());

            // delay
            $this->delay->setupEditAttributes();
            $this->delay->EditValue = $this->delay->AdvancedSearch->SearchValue;
            $this->delay->PlaceHolder = RemoveHtml($this->delay->caption());

            // delay_method
            $this->delay_method->setupEditAttributes();
            $this->delay_method->EditValue = $this->delay_method->options(true);
            $this->delay_method->PlaceHolder = RemoveHtml($this->delay_method->caption());

            // last_updated
            $this->last_updated->setupEditAttributes();
            if (!$this->last_updated->Raw) {
                $this->last_updated->AdvancedSearch->SearchValue = HtmlDecode($this->last_updated->AdvancedSearch->SearchValue);
            }
            $this->last_updated->EditValue = HtmlEncode($this->last_updated->AdvancedSearch->SearchValue);
            $this->last_updated->PlaceHolder = RemoveHtml($this->last_updated->caption());

            // user_level
            $curVal = trim(strval($this->user_level->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->user_level->AdvancedSearch->ViewValue = $this->user_level->lookupCacheOption($curVal);
            } else {
                $this->user_level->AdvancedSearch->ViewValue = $this->user_level->Lookup !== null && is_array($this->user_level->lookupOptions()) && count($this->user_level->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->user_level->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->user_level->EditValue = array_values($this->user_level->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $arwrk = explode(Config("MULTIPLE_OPTION_SEPARATOR"), $curVal);
                    $filterWrk = "";
                    foreach ($arwrk as $wrk) {
                        AddFilter($filterWrk, SearchFilter($this->user_level->Lookup->getTable()->Fields["userlevelid"]->searchExpression(), "=", trim($wrk), $this->user_level->Lookup->getTable()->Fields["userlevelid"]->searchDataType(), ""), "OR");
                    }
                }
                $sqlWrk = $this->user_level->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->user_level->EditValue = $arwrk;
            }
            $this->user_level->PlaceHolder = RemoveHtml($this->user_level->caption());

            // visible
            $this->_visible->setupEditAttributes();
            $this->_visible->EditValue = $this->_visible->options(true);
            $this->_visible->PlaceHolder = RemoveHtml($this->_visible->caption());

            // operator
            $this->operator->setupEditAttributes();
            $curVal = trim(strval($this->operator->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->operator->AdvancedSearch->ViewValue = $this->operator->lookupCacheOption($curVal);
            } else {
                $this->operator->AdvancedSearch->ViewValue = $this->operator->Lookup !== null && is_array($this->operator->lookupOptions()) && count($this->operator->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->operator->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->operator->EditValue = array_values($this->operator->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->operator->Lookup->getTable()->Fields["id"]->searchExpression(), "=", $this->operator->AdvancedSearch->SearchValue, $this->operator->Lookup->getTable()->Fields["id"]->searchDataType(), "");
                }
                $sqlWrk = $this->operator->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->operator->EditValue = $arwrk;
            }
            $this->operator->PlaceHolder = RemoveHtml($this->operator->caption());

            // datetime
            $this->datetime->setupEditAttributes();
            $this->datetime->EditValue = HtmlEncode(FormatDateTime(UnFormatDateTime($this->datetime->AdvancedSearch->SearchValue, $this->datetime->formatPattern()), $this->datetime->formatPattern()));
            $this->datetime->PlaceHolder = RemoveHtml($this->datetime->caption());

            // status
            $this->status->setupEditAttributes();
            $this->status->EditValue = $this->status->options(true);
            $this->status->PlaceHolder = RemoveHtml($this->status->caption());

            // order_by
            $this->order_by->setupEditAttributes();
            $this->order_by->EditValue = $this->order_by->AdvancedSearch->SearchValue;
            $this->order_by->PlaceHolder = RemoveHtml($this->order_by->caption());

            // url
            $this->url->setupEditAttributes();
            if (!$this->url->Raw) {
                $this->url->AdvancedSearch->SearchValue = HtmlDecode($this->url->AdvancedSearch->SearchValue);
            }
            $this->url->EditValue = HtmlEncode($this->url->AdvancedSearch->SearchValue);
            $this->url->PlaceHolder = RemoveHtml($this->url->caption());
        }
        if ($this->RowType == RowType::ADD || $this->RowType == RowType::EDIT || $this->RowType == RowType::SEARCH) { // Add/Edit/Search row
            $this->setupFieldTitles();
        }

        // Call Row Rendered event
        if ($this->RowType != RowType::AGGREGATEINIT) {
            $this->rowRendered();
        }
    }

    // Validate search
    protected function validateSearch()
    {
        // Check if validation required
        if (!Config("SERVER_VALIDATE")) {
            return true;
        }
        if (!CheckInteger($this->apcid->AdvancedSearch->SearchValue)) {
            $this->apcid->addErrorMessage($this->apcid->getErrorMessage(false));
        }
        if (!CheckDate($this->next_cycle->AdvancedSearch->SearchValue, $this->next_cycle->formatPattern())) {
            $this->next_cycle->addErrorMessage($this->next_cycle->getErrorMessage(false));
        }
        if (!CheckInteger($this->delay->AdvancedSearch->SearchValue)) {
            $this->delay->addErrorMessage($this->delay->getErrorMessage(false));
        }
        if (!CheckInteger($this->order_by->AdvancedSearch->SearchValue)) {
            $this->order_by->addErrorMessage($this->order_by->getErrorMessage(false));
        }

        // Return validate result
        $validateSearch = !$this->hasInvalidFields();

        // Call Form_CustomValidate event
        $formCustomError = "";
        $validateSearch = $validateSearch && $this->formCustomValidate($formCustomError);
        if ($formCustomError != "") {
            $this->setFailureMessage($formCustomError);
        }
        return $validateSearch;
    }

    // Load advanced search
    public function loadAdvancedSearch()
    {
        $this->apcid->AdvancedSearch->load();
        $this->grp->AdvancedSearch->load();
        $this->code->AdvancedSearch->load();
        $this->description->AdvancedSearch->load();
        $this->cron->AdvancedSearch->load();
        $this->next_cycle->AdvancedSearch->load();
        $this->delay->AdvancedSearch->load();
        $this->delay_method->AdvancedSearch->load();
        $this->last_updated->AdvancedSearch->load();
        $this->user_level->AdvancedSearch->load();
        $this->_visible->AdvancedSearch->load();
        $this->operator->AdvancedSearch->load();
        $this->datetime->AdvancedSearch->load();
        $this->status->AdvancedSearch->load();
        $this->order_by->AdvancedSearch->load();
        $this->url->AdvancedSearch->load();
    }

    // Set up Breadcrumb
    protected function setupBreadcrumb()
    {
        global $Breadcrumb, $Language;
        $Breadcrumb = new Breadcrumb("db2");
        $url = CurrentUrl();
        $Breadcrumb->add("list", $this->TableVar, $this->addMasterUrl("apclist"), "", $this->TableVar, true);
        $pageId = "search";
        $Breadcrumb->add("search", $pageId, $url);
    }

    // Setup lookup options
    public function setupLookupOptions($fld)
    {
        if ($fld->Lookup && $fld->Lookup->Options === null) {
            // Get default connection and filter
            $conn = $this->getConnection();
            $lookupFilter = "";

            // No need to check any more
            $fld->Lookup->Options = [];

            // Set up lookup SQL and connection
            switch ($fld->FieldVar) {
                case "x_grp":
                    $lookupFilter = $fld->getSelectFilter(); // PHP
                    break;
                case "x_cron":
                    break;
                case "x_delay_method":
                    break;
                case "x_user_level":
                    break;
                case "x__visible":
                    break;
                case "x_operator":
                    break;
                case "x_status":
                    break;
                default:
                    $lookupFilter = "";
                    break;
            }

            // Always call to Lookup->getSql so that user can setup Lookup->Options in Lookup_Selecting server event
            $sql = $fld->Lookup->getSql(false, "", $lookupFilter, $this);

            // Set up lookup cache
            if (!$fld->hasLookupOptions() && $fld->UseLookupCache && $sql != "" && count($fld->Lookup->Options) == 0 && count($fld->Lookup->FilterFields) == 0) {
                $totalCnt = $this->getRecordCount($sql, $conn);
                if ($totalCnt > $fld->LookupCacheCount) { // Total count > cache count, do not cache
                    return;
                }
                $rows = $conn->executeQuery($sql)->fetchAll();
                $ar = [];
                foreach ($rows as $row) {
                    $row = $fld->Lookup->renderViewRow($row, Container($fld->Lookup->LinkTable));
                    $key = $row["lf"];
                    if (IsFloatType($fld->Type)) { // Handle float field
                        $key = (float)$key;
                    }
                    $ar[strval($key)] = $row;
                }
                $fld->Lookup->Options = $ar;
            }
        }
    }

    // Page Load event
    public function pageLoad()
    {
        //Log("Page Load");
    }

    // Page Unload event
    public function pageUnload()
    {
        //Log("Page Unload");
    }

    // Page Redirecting event
    public function pageRedirecting(&$url)
    {
        // Example:
        //$url = "your URL";
    }

    // Message Showing event
    // $type = ''|'success'|'failure'|'warning'
    public function messageShowing(&$msg, $type)
    {
        if ($type == "success") {
            //$msg = "your success message";
        } elseif ($type == "failure") {
            //$msg = "your failure message";
        } elseif ($type == "warning") {
            //$msg = "your warning message";
        } else {
            //$msg = "your message";
        }
    }

    // Page Render event
    public function pageRender()
    {
        //Log("Page Render");
    }

    // Page Data Rendering event
    public function pageDataRendering(&$header)
    {
        // Example:
        //$header = "your header";
    }

    // Page Data Rendered event
    public function pageDataRendered(&$footer)
    {
        // Example:
        //$footer = "your footer";
    }

    // Page Breaking event
    public function pageBreaking(&$break, &$content)
    {
        // Example:
        //$break = false; // Skip page break, or
        //$content = "<div style=\"break-after:page;\"></div>"; // Modify page break content
    }

    // Form Custom Validate event
    public function formCustomValidate(&$customError)
    {
        // Return error message in $customError
        return true;
    }
}
